/* This module implements short-term national targets to 2020 and 2050 as pledged following the COP15 Copenhagen deal.
 * It precedes "shares" module which fills in the national targets after this period
 * extended 2016 to include Paris INDCs
 */
package jcm.mod.regemit;

import java.util.*;
import jcm.core.*;
import jcm.core.cur.*;
import jcm.core.par.param;
import jcm.core.ob.module;
import jcm.core.reg.*;
import jcm.core.ob.loopcalc;
import jcm.mod.luc.CalcLucEmit;
import jcm.mod.ogas.othgasemit;
import jcm.mod.socio.popgdp;
import static jcm.core.complexity.*;
import static jcm.core.report.*;
import static jcm.gui.gen.colfont.*;
import static jcm.mod.regemit.pledges.kyoto_opts.*;
import static jcm.mod.regemit.pledges.gdpopts.*;
import static jcm.mod.regemit.pledges.pledgelevel.*;
import static jcm.mod.regemit.pledges.targettype.*;
import static jcm.mod.regemit.pledges.gasinc.*;
import static jcm.mod.regemit.pledges.sectorinc.*;
import static jcm.mod.regemit.pledges.UNEPinfo.*;
import static jcm.mod.regemit.pledges.lucinfo.*;

public class pledges extends module {
	
    //============== PARAMS ==========
	public  enum pledgelevel { unconditional, conditional, mixed  }

    public enum kyoto_opts {  real_emissions, Kyoto_targets_excUSA, Kyoto_targets_excUSA_prolonged, Kyoto_targets_incUSA }
 
    public enum gdpopts { GDP_PPP, GDP_MER }

    /* simplify options by offer in one menu what combis pledges make sense ? 
     * 1235, 235, 23, 2, 3, 5, 35 
     * or maybe simpler (if takes more space) to have a yes/no option for each group (Kyoto, 2020, 2030, 2050) - already got three! 
     * but does it make sense to implement later pledges without earlier (exc history) ? 
     * yes - maybe set real emissions in 2020 (exc hotair) to examine 2030 pledges ... but then start line from when... - 2020? 
     */
    public enum pledgeopts {
    	all_inc_Kyoto, 			
    	all_future_pledges, 	 
    	all_future_belowbase,   
    	} 
    
    public param usepledges = new param("apply pledges", true);
    
    public param beyond2030 = new param("apply 2050", false, expert); 
    public param EU2050 = new param("EU targets 2050", false, expert);
    
    public param<pledgelevel> level = new param("pledge level", pledgelevel.values(), mixed);
    public param<kyoto_opts> kyoto = new param("Kyoto Protocol", kyoto_opts.values(), real_emissions); //, Kyoto_targets_excUSA_prolonged);
    
    public param sephotair = new param("Separate AAUsurplus", true);
    public param Ax1only = new param("AAUsurplus only Ax1", false); //removes surplus from EEurope and F.Soviet Union, but not from India, China etc  
    public param AAUfrac = new param("fraction surplus AAUs used", 0, 0, 100, "percent"); //note set to zero by default!
    
    public param<gdpopts> choosegdp = new param("gdp for intensity", gdpopts.values(), GDP_PPP, expert);
    
    public param UNEPbase = new param("use UNEP Gap baselines", false, expert); //mainly affects mexico and SKorea in 2020
    public param sepLUC = new param("separate LUC", true);
  

// ============ VARIABLES=========
    public Map<region, Integer> endtarget = new HashMap(50); //stores end year of target
    public Set<region> notarget = new LinkedHashSet(50); //countries with no target
    Set<region> EU = new LinkedHashSet(27);
    public Map<region, List<target>> targets = new LinkedHashMap();
    emitquota eq;
    emitbase eb;
    boolean fixedEU=false;
     curveset efb, efq, hfos, gdp_mer, gdp_ppp;

       //==================== INTERACTIONS ===========
    //
    public void initsetup() {
        eq = gm(emitquota.class);
        eb = gm(emitbase.class);
        follows(gm(emitbase.class));
        follows(gm(AviaShipEmit.class).emitbase);
        setaffectedby(gm(shares.class)); //not efficient - in principle pledges part shouldn't change unless shares over-rides pledges - but it can
        EU=regman.allreg.findreg("EU28").subreg(regman.nations);
        settargets();
    }

    public void setinteractions() {
        setaffectedby(level, usepledges.istrue());
        setaffectedby(choosegdp, usepledges.istrue());
        setaffectedby(beyond2030, usepledges.istrue());
        setaffectedby(sephotair, (usepledges.istrue() || !(kyoto.chosen==real_emissions)));
        setaffectedby(Ax1only, sephotair.istrue() && (usepledges.istrue() || !(kyoto.chosen==real_emissions)));
        setaffectedby(AAUfrac,  (usepledges.istrue() || !(kyoto.chosen==real_emissions)) && sephotair.istrue());
        setaffectedby(EU2050, beyond2030.istrue() && usepledges.istrue());
        setaffectedby(UNEPbase, usepledges.istrue());
        setaffectedby(sepLUC, usepledges.istrue());
    }
     //**************** LOOP METHODS *********
    public void precalc() {

        efb =eb.emitfosbase_nat;
        efq= eq.emitfos_nat;
        gdp_mer = gm(popgdp.class).gdp_mer_nat;
        gdp_ppp = gm(popgdp.class).gdp_ppp_nat;
        hfos = histemitdata.fosCO2; //beware in kilo-ton not mega-ton!
        
        if (!fixedEU) fixEU(); //normalise total to 20% low and 30% high  -depends on hfos
        for (region r : EU) endtarget.put(r,  EU2050.istrue() ? 2050 : 2030);
        
        applytargets();
        /*apply targets can run in precalc only presuming no dependence on other modules within calcstep loop?
        eg popgdp - but that also works entirely in precalc
        and emitbase - also works in precalc except totals and bunker
        */
        fillhistory();  
        /* fill history is for countries without any target
       in principle shouldn't need to do fill history more than once - check if can do in initsetup - but beware changing start year in shares etc.
        also might want option to remove all without targets - should be an option in shares?
        */
         }


    //==================== TARGETS  ===========
    /*use a class to group target info together (not an enum, because targets may need to be user-adjustable)
     *may need several targets for different years, for each country
     *may have targets referring to a group of countries - put first, can be overridden for specific countries
     *some countries may have multiple targets for different sectors in same year, or different types different years  
     */

    /* Code Structure improvements:
     * not using region enums as below for UNEP, LUC  constructor is better struc than st() method below.
     * so convert all into classes like target, then can share common methods...
     * eventually - neater if all regions would be enums ... replace regions.txt
     * target would be neater as a scala case class
     * but scala has no simple enums 
     */

    public enum targettype { absreduc, intensity, baseline, fixnum }
    public enum sectorinc { allsec, industry, LUC }
    public enum gasinc { K7gas, K6gas, co2 } 
    //note 6gas is Kyoto, 7th gas is NF3 (not yet included in JCM). Only recent targets can incl K7

    public class target {
        public region reg;
        public int baseyear,  targetyear;
        public targettype type; public sectorinc sec; public gasinc gases;
        public float low,  high,  likely; //% reductions
        public float baseemit;
        target(region r, targettype t, sectorinc s, gasinc g, int b, int y, float l, float h, float v) {
            reg = r;
            type = t; sec=s; gases=g;
            baseyear = b;
            targetyear = y;
            low = l;
            high = h;
            likely = v;
            //deb("target"+r+" "+t+" "+s+" "+g+" "+b+" "+y+" "+l+" "+h+" "+v);
        }

    } // class target

    void st(String regname, Object ... info) { 
    	region r = regman.allreg.findreg(regname);
    	if (r == null) { deb("Targets error: no region " + regname);    return; }
    	st (r, info); 
    }
    
    void st(region r, Object ... info) {
        
        List<target> lt = (!targets.containsKey(r) ? new ArrayList(4) : targets.get(r));

        int baseyear = 0, targetyear = 0;
        boolean gotyear = false;
        int count = 0;
        float low = 0, high = 0, value = 0;
        targettype type =absreduc; sectorinc sec =allsec; gasinc gases=K6gas; //note defaults 
        
        for (Object o : info) {

        	//if got some target % info, but encounter diff type info, must be start new target, so save current info and reset before continue
        	if (count>0 && !((o instanceof Number) && ( ((Number) o ).intValue() < 1000)   )) {	
				lt.add(new target(r, type, sec, gases, baseyear, targetyear, low, high, value));
				count = 0; gotyear=false;         		
        	}
        	
        	if (o instanceof targettype) type=(targettype)o;
        	if (o instanceof sectorinc) sec=(sectorinc)o;
        	if (o instanceof gasinc) gases=(gasinc)o;
        	
        	
        	if (o instanceof Number ) {
        		Number j = (Number) o;	
        		if (j.intValue() > 1000) /*it's a year*/ {
        			if (gotyear) baseyear = targetyear;
        			targetyear = j.intValue();
        			gotyear = true;
        		} else { //high, low, mid target (%)
        			gotyear = false;
        			count++;
        			if (count == 1) {
        				low = j.floatValue();
        				high = low;
        				value = low;
        			}
        			if (count == 2) {
        				high = j.floatValue();
        				value = (low + high) / 2;
        			}
        			if (count == 3) {
        				value = j.floatValue();
        			}
        		}
        	} //number
        } //o
        
        lt.add(new target(r, type, sec, gases, baseyear, targetyear, low, high, value)); //save last one
        endtarget.put(r, targetyear);

        if (regman.nations.contains(r)) {
            targets.put(r, lt);
            return;
        }
        //if it's a region-set apply to all nations within region
        for (region sr : r.subreg(regman.nations)) targets.put(sr, lt);

    } //end st

    
  //========== SET TARGETS ==========
    //note: set targets for groups, followed by nations within them, the latter will over-ride group target
   
    void settargets() {
    	//for orig info check docs from UNFCCC/ CAIT http://cait.wri.org/indc/
        
    	//=======================================================
    	//the first group -Kyoto AxB- all have economy-wide, 7gas targets for 2030 (or 25)
    	st("USA", 1990, 2010, -7, 2005, 2020, -17, 2025, -26, -28, -28, 2050, -80, -85); //no 2030 target in INDC, 80% or more by 2050 (but is that from 2005??) 
    	//st("USA", 1990, 2010, -7, 2005, 2020, -17, 2025, -30, 2030, -42, 2050, -83); //old from copenhagen - had a 2030 target too
    	st("Australia", 1990, 2010, +8, 2000, 2020, -5,  -25, -15, 2005, 2030, -26, -28); 
        st("New Zealand", 1990, 2010, 0, 2020, -10, -20, 2005, 2030, -30); //NZ INDC notes 2030 target is 11% below 1990 - note dominated by agriculture, as elec already 80% renewable
        st("Canada", 1990, 2010, -6, 2005, 2020, -17, 2030, -30, 2006, 2050, -60, -70); //CAN - 2050 targets are from pre-copenhagen wrt 2006
        st("Japan", 1990, 2010, -6, 2020, -15, -25, 2013, 2030, -26); //Japan INDC says 2030 ~-25.4% below 2005, or 1.042 Gt-CO2eq 
        
        st("Russia", 1990, 2010, 0, 2020, -15, -25, 2030, -25, -30); //note lower pledge was -22 pre Cop?
        st("Ukraine", 1990, 2010, 0, 2020, -15, -25, 2030, -40); //check 2020??
        st("Belarus", 1990, 2010, -8, 2020, -5, -10, 2030, -28); 
        st("Kazakhstan", 1992, 2020, -15, 1990, 2030, -15, -25 ); //6gas

        st("Switzerland", 1990, 2010, -8, 2020, -20, -30, 2025, -35, 2030, -50); //check 2020? //includes mex credits //longterm plan  1–1.5 tCO2eq/capita 
        st("Norway", 1990, 2010, +1, 2020, -30, -40, 2030, -40); //within EU sharing //recall copenhagen original annoucement -40, the -30 was added jan 2010 in response to low ambition of others
        st("Iceland", 1990, 2010, +10, 2020, -30, 2030, -40); //within EU-ETS
        
        //=======================================================
        //st("EU28", 1990,  2010, -8, 2020, -20, -30, 2030, -40); //jointly
        //instead of obove, continue using 2020 BS agreement as below, scaled to above total targets - fixed in method fixEU
        //otherwise looks odd due to very different post-1990 history

        //st("EU27", 1990,  2010, -8, 2020, -20, -30, 2050, -80, -90); 
        //figures for EU15 for 1990 are from EU burden sharing agreement, for the 10 eastern countries they are from Kyoto protocol - note 5 have different base years
        //figures for 2020 below are taken from  + EU effort-sharing agreement for -20%apr09 http://eur-lex.europa.eu/LexUriServ/LexUriServ.do?uri=OJ:L:2009:140:0136:0148:EN:PDF ANNEX II
        //note normalisation to 20% total reduction in fixEU method
        
        st("Austria", 1990, 2010, -13, 2005, 2020, -16);
        st("Belgium", 1990, 2010, -7.5, 2005, 2020, -15);
        st("Bulgaria", 1988, 2010, -8, 2005, 2020, +20);
        st("Cyprus", 2005, 2020, -5);
        st("Czech Republic", 1990, 2010, -8, 2005, 2020, +9);
        st("Denmark", 1990, 2010, -21, 2005, 2020, -20);
        st("Estonia", 1990, 2010, -8, 2005, 2020, +11);
        st("Finland", 1990, 2010, 0, 2005, 2020, -16);
        st("France", 1990, 2010, 0, 2005, 2020, -14);
        st("Germany", 1990, 2010, -21, 2005, 2020, -14);
        st("Greece", 1990, 2010, +25, 2005, 2020, -4);
        st("Hungary", 1986, 2010, -6, 2005, 2020, +10); //technically, base year for Kyoto is average of 85-87
        st("Ireland", 1990, 2010, +13, 2005, 2020, -20);
        st("Italy", 1990, 2010, -6.5, 2005, 2020, -13);
        st("Latvia", 1990, 2010, -8, 2005, 2020, +17);
        st("Lithuania", 1990, 2010, -8, 2005, 2020, +15);
        st("Luxembourg", 1990, 2010, -28, 2005, 2020, -20);
        st("Malta", 2005, 2020, +5);
        st("Netherlands", 1990, 2010, -6, 2005, 2020, -16);
        st("Poland", 1988, 2010, -6, 2005, 2020, +14);
        st("Portugal", 1990, 2010, +27, 2005, 2020, +1);
        st("Romania", 1989, 2010, -8, 2005, 2020, +19);
        st("Slovenia", 1986, 2010, -8, 2005, 2020, +4);
        st("Slovakia", 1990, 2010, -8, 2005, 2020, +13);
        st("Spain", 1990, 2010, +15, 2005, 2020, -10);
        st("Sweden", 1990, 2010, +4, 2005, 2020, -17);
        st("United Kingdom", 1990, 2010, -12.5, 2005, 2020, -16); //using EU effort-sharing for -20%
        //st("United Kingdom", 1990,  2010, -12.5, 2020, -34, -42, 2050, -80); //uk domestic target -34% by 2020, -80% by 2050

        st("Croatia", 1990, 2010, -5, 2005, 2020, -5); //CHECK IF 2020 NUMBER CORRECT?
     	
        
        //=======================================================
        st("China", intensity, 2005, 2020, -40, -45, 2030, -60, -65);
        //China also targets  % renewables share (2020 15%, 2030 20%) and forest stocks (2020 +1.3bn m3  2030 +4.5bn m3) all wrt 2005
        //INDC peaking by 2030, effort earlier
        //INDC includes summary achievements by 2014 - intensity -34%, renewables 11.2%, forest stocks +2.2bn m3), huge increase solar wind,  some hydro too
        
        
        st("India", intensity, 2005, 2020, -20, -25, 2030, -33, -35); 
        //pre-cop15 India said -24 by 2020, -37 by 2030, but latter was not retained in cop-accord
        //INDC 2030 also specifies: 40% elec renewable, 2.5-3.0GtCO2eq extra forest sink
        
        st("Malaysia", intensity, 2005, 2030, -35, -45); 
        //CO2, CH4, N2O, all sectors
        //base year 2005, emissions 288,663 Gg CO2eq (of which LUC=25,667), GDP in the base year (constant price at 2005): RM 543.578 billion
     
        
        //=======================================================
        //st("Brazil", baseline, 2020, -36.1, -38.9, absreduc, 2005, 2025, -37, 2030, -43);
        /*Brazil's INDC covers all 6 gases and is economy-wide, so it includes a large reduction in LUC which already happened post 2005
         * using AR5 GWPs,  equiv numbers without accounting for conservation areas / indigenous lands would be 31, 37
         * INDC gives some info: based on 2.1 GtCO2e in 2005, (old nvc 2.0 2020 GWPSAR)  1.3 2025, 1.2 2030
         * 
         * Emissions (all gas/sec) already fell 41% 2005-2012 (SAR GWPs) - or 52% 2004-12 (AR5 GWPs)
         * Brazil's 2020 pledge is defined according to specific reductions in each sector, the baseline figure is derived from that
         * Brazil old pledge info: LUC+agric 2005 1540Mt, uncondit pledge 2020 909Mt  
         * So in 2005 if (fo+la) = 2.1 and la= 1.54 => fo = 0.56 
         * in 2025 (f+l) = 1.3, as la(20) = 0.91,  => la(25) = 1.54 - (1.54-0.91)*20/15 = 0.7 => fo = 0.6 = +7% (2025/2005) 
         * Brazil 2014 doc says 
         * 2020 target means (Mt CO2e) 634 Energy, 234 Ind/Waste, 596 Agric, 516 LUC (TOT 1980)
         * 2005 equiv nums ~ 350, 125, 420, 1250 (TOT 2145) (and note peak LUC was 2004, ~1700, also 1995 ~1900) 
         * as other sectors change little 2004-2005, total 2004 is about 500 higher (2645/2145) => 2012 / 2005 = (1-52%) *2645/2145 = 1- 41% (consist above)        
		 * => for just fossil (~en+0.5*in) 2020 is +84% !!!   (they note 2005-12 is +39% - consistent)    
		 * maybe fairest is to put abs reduc wrt 2012 figs given (since LUC then low, not much further to fall ): 2012  1270, 2020 1980, 2025 1351, 2030 1223)
		 * the number for 2020 will be overridden anyway if luc+unep option used (see futureluc skip method)    
        */ 
        st("Brazil", baseline, 2020, -36.1, -38.9,  absreduc, 2012, 2025, +6, 2030, -4 ); 
        
         
        st("Indonesia", baseline, 2020, -26, -41, 2030, -29, -41);
        //st("Indonesia", baseline, 2020, -26, -41, absreduc, 2005, 2030, +13.6, -5.5); 
        /*Indonesia includes CO2, CH4, N2O, all (5) sectors
        around 2005 (?) 63% LUC+peat+fires  , 19% fossil (63% of 1800 is  1134 - but lucpledge info below gives 1610 - it was very variable)  
        2030 BAU = 2.881 GtCO2eq, also 2000 1400, 2005 1800, => 2030 BAU is +60% wrt 2005 => absreduc targets are +13.6% and -5.5%
        this still squeezes the fossil too much - as we assume LUC reduces more 
        if fossil 2005 is 345 (JCM - consistent 19%) and LUC/Ogas the rest (dealt with other modules)
        if by 2030 luc drops to about 620 (JCM), and total is 2045 (-29%) => rest has 1425, if half that is fossil its 712, still +106% wrt 2005...     
       */
        
        //=========================================================
        st("Mexico", baseline, 2020, -30, 2030, -22, -36, absreduc, 2000, 2050, -50); 
        /* Mexico 22%-36% is for GHG (only) - additionally  -51%-70% Black Carbon aerosol reduction
        INDC says implies a net emissions peak starting from 2026
        Baseline numbers: 2020: 906 MtCO2e (792 GHG and 114 BC / 127,177 metric tons) 2025: 1013 MtCO2e (888 GHG and 125 BC / 138,489 metric tons) 2030: 1110 MtCO2e (973 GHG and 137 BC / 152,332 metric tons)
        emissions intensity per unit of GDP will reduce by around 40% (unconditional) from 2013 to 2030.
        The -50% is mandated by LGCC 
        Mexico's 2030 INDC seems less ambitious than its 2020 pledge! but maybe quite different baseline  - this one starts from 2013
        For Mexico, the original plan for 2020 was quite complex, and specified in GtC for each sector rather than %, hard to interpret simply.
        //OLD st("Mexico", 2002, 2012, -8, 2050, -50); //check Special Programme on CC
         */
        st("South Korea", baseline, 2020, -21, -30, 2030, -37);
        //includes all gases, all sectors EXCEPT LUC (tbd later) 
        //BAU 2020 782.5 2025 809.7 2030 850.6 MtCO2eq
        //st("Korea", 2005, 2020, -4); //widely cited intrepretation of 2020 pledge
        st("Singapore", baseline, 2020, -16, intensity, 2005, 2030, -36); //see Sustainable Singapore Blueprint in April 2009
        //Singapore’s greenhouse gas (GHG) emissions per S$GDP (at 2010 prices) in 2005 is 0.176 kgCO2e/S$.
        //GDP (2005) at 2010 prices: S$232.77 billion (Source: Department of Statistics, Singapore)
        
        st("Thailand", baseline, 2030, -20, -25 );
        st("Vietnam", baseline, 2030, -8 -25);
        st("Bangladesh", baseline, 2030,  -5, -15 );
        //Bdesh 2030 total BAU 234MtCO2e
        
        st("Iran", baseline, 2030, -4, -12);
        //compared to baseline calculated from 2010
        st("Turkey", baseline, 2030,-21);
        //see plot at end - deviation from 2012: 430, 2015: 449/77, 2020: 599/673, 2030: 929/1175  
        
        st("South Africa", baseline, 2020, -34, 2025, -42,  absreduc, 2020, 2030, 0);
        //this doesn't seem to be working - looking at the plot
        /*South Africa’s emissions by 2025 and 2030 will be in a range between 398 and 614 Mt CO2–eq
         *peak between 2020 and 2025, plateau for approximately a decade and decline in absolute terms thereafter. 
         *Values for key years are specified in the NCCRP.
         */
        st("Nigeria", baseline, 2030, -20, -45);
        //Nigeria estimated Emit/capita current ~2tCO2e, 2030base ~3.4, 2030condit ~2
        st("Congo Democratic Republic", baseline, 2000, 2030, -17);
        //Congo assumed base emissions 2030: 430 Mt CO2e - note big jump up compared to present (~240 in 2012)
        st("Angola", baseline, 2030, -35, -50);
        //Angola assumed base emissions 2030: 193.25 MtCO2e
        
        st("Argentina", baseline, 2030, -15, -30);
        //baseline starts deviate 2005, so we are already ~9% below it 
        st("Venezuela", baseline, 2030, -20);
        //smooth-growth curve baseline deviating from 2012
        st("Colombia", baseline, 2030, -20, -30); //base deviates form 2015

        
        notarget.addAll(regman.nations.reg);
        notarget.removeAll(targets.keySet());
        
        for (region r : notarget) {
        	st(r, baseline, 2020, -5, 2030, -10, -20); //default for countries not listed or INDC not quantitative 
        }
        notarget.clear();
        

    } //end settargets

    //Pakistan - they're still working on the peaking data
    //Egypt - no target - portfolio measures
    //Iraq - baseline target, but only in arabic, plot unclear - note variable history due wars => baseline jumps up 
    //Saudi Arabia - INDC absolute reduction from baseline(s) - mitigation co-benefits ambitions of up to 130 million tons of CO2eq avoided by 2030 annually 
    //UAE - INDC portfolio measures, 24% clean energy ...
    //Kuwait - actions only
    //Quatar - efficiency / renewables actions
    //Uzebekistan - no INDC yet

    

    //specific data for countries, from UNEP Emissions Gap report appendix graphics  in MtCO2eq,- excluding LUCF for Ax1, including LUCF for rest
    public enum UNEPinfo { te90, te05, te20BAU, te20UP, te20CP}
    public enum UNEPpledge {
       Australia(415, 525, 645, 470, 370),
       Brazil(1400, 1942, 2703, 1728, 1651), //note - adjusted numbers consistent with Brazilian info - UNEP graphic suggests 1600, 2400, 2700, 1700, 1600
       Canada(595, 725, 805, 805, 605),
       China(3700, 7200, 12600, 12300, 11500),
       EU27(5550, 5100, 5450, 4475, 3400),
       India(1250, 2050, 3450, 3550, 3550),
       Indonesia(700, 2100, 2800, 2000, 1700),
       Japan(1270, 1360, 1305, 1305, 950),
       Mexico(500, 720, 885, 825, 620),
       Russia(3375, 2100, 2500, 2850, 2500),
       South_Africa(325, 490, 750, 665, 500),
       South_Korea(300, 595, 810, 570, 570),
       USA(6075, 7100, 7325, 7325, 5900);
       
       
       Map<UNEPinfo, Integer> info=new EnumMap(UNEPinfo.class);
       UNEPpledge(int... ff) { int i=0; for (int f : ff) { info.put(UNEPinfo.values()[i], f); i++; } }
       public int get(UNEPinfo p) {return info.get(p); }
       public static UNEPpledge find(region r) { try { return valueOf(r.getName().replace(" ", "_"));  } catch (Exception e) { return null; }}

   }

    //data on LUC for the biggest LUC countries
    public enum lucinfo { lucbase, luc05, luc20BAU, luc20UP, luc20CP}
    public enum LUCpledge {
    	
  
       Brazil(818+369, 1060+480, 1084+627, 415+494, 415+461), //CenariosparaOfertaBrasilieradeMitiga.pdf - base=1994, adding Uso da Terra + Agropecuaria
       Indonesia(0, 840+770, 730+890, 1074, 759/*2030: 670+970 */ ), //Indonesia_ghg_cost_curve_english.pdf, adding LULUCF + Peat, Exhibit 2 (history), for future assume 75% of redn is from LULUCF
       //note also corrections for Brazil and Indonesia in CalcLucEmit module
       ;
       
       Map<lucinfo, Integer> info=new EnumMap(lucinfo.class);
       LUCpledge(int... ff) { int i=0; for (int f : ff) { info.put(lucinfo.values()[i], f); i++; } }
       public int get(lucinfo p) {return info.get(p); }
       public static LUCpledge find(region r) { try { return valueOf(r.getName().replace(" ", "_"));  } catch (Exception e) { return null; }}
   }

    //================= APPLY TARGETS =================
    public void applytargets() {
       //note efb and efq are in megaton carbon, whereas history and target in kiloton - hence the 1000f factors
        //note changed jun2011 to use gdp_nat - previously was regional = bug if regionset not separating india and china?
        curveset gdp = (choosegdp.chosen == GDP_PPP) ? gdp_ppp: gdp_mer;

        for (int y=1990; y<=2030; y++) { efq.set("AAUsurplus", y, 0); }

        for (region r : targets.keySet()) {
            //if (!regman.nations.contains(r)) { deb("Error: can only apply targets to nations, not to "+r.name); return; }
            int py = 1990;
            float target, bau, tarlin=0,  tbase;

            //go through targets (earlier first) without resetting  year
            for (target t : targets.get(r)) {
                //note the assumption is that 2005 targets  apply to actual emissions not to kyoto levels
                
                if (t.targetyear ==2020 && sepLUC.istrue() && LUCpledge.find(r)!=null) {
                    //separate targets for LUC and fossil CO2, don't apply same ratio reduction for both
                    //first get LUC info from national plan
                    //lucbase, luc05, luc20BAU, luc20UP, luc20CP

                    float LUCtarget=(level.chosen==conditional? LUCpledge.find(r).get(luc20CP) : /*level.chosen==unconditional ? */LUCpledge.find(r).get(luc20UP) /*: t.likely*/);
                    float tottarget=(level.chosen==conditional? UNEPpledge.find(r).get(te20CP) : /*level.chosen==unconditional ? */UNEPpledge.find(r).get(te20UP) /*: t.likely*/);
                    /* assume fossil LUC scales to the rest  - although much of the rest is CH4, N2O !
                     why does this give an absolute reduction, <2005?  **because UNEP in 2005 much too high - maybe due LUC **

                     checked numbers UNEP consistent with brazil cénariosoferta doc
                      checked JCM CO2eq consistent with brazil cénariosoferta doc for 2005, and JCM CO2fos consistent with BP... (
                     
                     */
                    target=(tottarget-LUCtarget)  * efb.get(r,2005) / (UNEPpledge.find(r).get(te05) - LUCpledge.find(r).get(luc05) );

                    //apply LUCtarget to LUC
                    float luc_jcm_09=gm(CalcLucEmit.class).lucemit.get(r, 2009)/1000f, luc_jcm_05=gm(CalcLucEmit.class).lucemit.get(r, 2005)/1000f;
                    //normalize to 2005 to remove the other gases from agropecuaria etc., also convert units tCO2=> tC
                    float LUCtarget_adj=LUCtarget* luc_jcm_05 / LUCpledge.find(r).get(luc05);
                    for (int y=2010; y<=2020; y++) {
                        eq.emitluc.set(r, y, (luc_jcm_09*(2020-y) + (LUCtarget_adj)*(y-2009))/11f  );
                    }
                    
                    
                    // tbase= ((UNEPpledge.find(r).get(te20BAU) - LUCpledge.find(r).get(luc20BAU)) / (UNEPpledge.find(r).get(te05) - LUCpledge.find(r).get(luc05)))* efb.get(r, 2005);

                } else { //not sepLUC

                	float base = hfos.get(r, t.baseyear) / 1000f;
                	float trat=(100f + (level.chosen==conditional? t.high : level.chosen==unconditional ? t.low : t.likely)) / 100f;

                	if (t.targetyear ==2020 && t.type == baseline && UNEPbase.istrue() && UNEPpledge.find(r)!=null ) {
                		//only applies to devg countries with data inc LULUCF, pledge 2020;
                		int ty=t.targetyear;
                		//e90, te05, te20BAU, te20UP, te20CP

                		tbase= (UNEPpledge.find(r).get(te20BAU)/ UNEPpledge.find(r).get(te05))* efb.get(r, 2005);
                		//* (12f / 44f) * (efb.get(r, ty)* (44f / 12f) / eb.equiv(r, ty) );
                		//deb("applying UNEP base for"+ r.getName()+ " "+efb.get(r, ty)+" "+eb.equiv(r, ty) );

                	}
                	else tbase=
                			t.type == baseline ? efb.get(r, t.targetyear)
                					: base * (t.type == intensity ? gdp.get(r, t.targetyear) / gdp.get(r, t.baseyear) : 1f);
                			//deb("pledges "+r.name+" "+t.targetyear+" "+t.baseyear+" tbase= "+tbase);
                			target = trat * tbase;

                } //not sepLUC
                
                //start by setting to real emissions up to 2008 (if not done already)
                while (py <= 2007) {
                    efq.set(r, py, hfos.get(r, py) / 1000f);
                    py++;
                }

                //kyoto targets 2008-12
                if (t.targetyear == 2010 && kyoto.chosen!=real_emissions && (kyoto.chosen==Kyoto_targets_incUSA || !r.name.equals("USA"))) {
                        while (py <= 2012) {
                            bau=hfos.get(r, py) / 1000f;
                            if (target>bau && sephotair.istrue()&& py<=2030) {
                                efq.set("AAUsurplus", py, efq.get("AAUsurplus", py)+(target-bau)*(float)AAUfrac.getval()/100f);
                                efq.set(r, py, bau);
                            }
                            else  efq.set(r, py, target);
                             py++;
                        }
                        tarlin = (kyoto.chosen==Kyoto_targets_excUSA_prolonged) ? target : hfos.get(r, 2012) / 1000f;
                }
                //else continue with real emissions up to 2012
                //May 2016 changed 2012 to 2015
                else while (py <= 2015) {
                    efq.set(r, py, hfos.get(r, py) / 1000f);
                    if (py==2015) tarlin= hfos.get(r, py) / 1000f;
                    py++;
                }

                //apply linear approach towards 2020, 2030 target
                while (py <= t.targetyear) {
                	
                    bau=efb.get(r, py);
                    tarlin=(tarlin * (t.targetyear - py) + target) / (1 + t.targetyear - py);
                    if (tarlin>bau && sephotair.istrue() && py<=2030 && (!Ax1only.istrue() ||regman.allreg.findreg("AXB").contains(r))) {
                        //deb("year="+year+" region="+r+ "surp=" + (tarlin-bau));
                        efq.set("AAUsurplus", py,  efq.get("AAUsurplus", py)+(tarlin-bau)*(float)AAUfrac.getval()/100f);
                        efq.set(r, py, bau);
                    }
                    else efq.set(r, py, tarlin);
                    //deb("year="+year+" region="+r+ "t.baseyear="+t.baseyear);
                    if (Float.isNaN(efq.get(r, py))) { efq.set(r, py, 0); }//if (py % 5 ==0) deb("problem target for "+r.name+" in "+py);  }
                    py++;
                }
            } //target
        //while (year<=2050) {fosCO2_nat.set(r, year, curve.dud); year++; } // old code to fill in blanks for reset after remove target??
            //deb("target for "+r.name+" in "+py+" is "+efq.get(r, py));
        } //region
         for (int y=1990; y<=2030; y++) { //copy from nat to region etc
             float surplus= efq.get("AAUsurplus", y);
             eq.emitfos.set("AAUsurplus", y, surplus);
             eq.emitfos_unitsCO2.set("AAUsurplus", y, surplus*44f/12f); 
             eq.emitequiv.set("AAUsurplus", y, surplus*44f/12f); 
             eq.emitequiv_nat.set("AAUsurplus", y, surplus*44f/12f); 
         }
    }

    //===================== FIX EU ================
// fix EU, scaling as a group - necessary because burden sharing agreement is only for domestic part not for large industry covered by EU-ETS, so original numbers don't add to -20%
    void fixEU() {
    float tot2020=0, totbase=0;
        for (region r : EU) {
            List<target> tt=targets.get(r);
            target t=tt.get(tt.size()-1); // the last target - as not all have a Kyoto target
            tot2020+=hfos.get(r,t.baseyear)  *(100f+t.low)/100f; //note baseyear for EUBS is 2005
            totbase+=hfos.get(r,1990)  ; //but group target is wrt 1990...
        }
        float fac=totbase/tot2020; //fac to convert 2020 back to 1990 normalised on average for group 

        for (region r : EU) {
            List<target> tt=targets.get(r);
            target t=tt.get(tt.size()-1); // the last target - as not all have a Kyoto target
            float tlo=100f+t.low, thi=100f + t.high; // why this - as no diff lo, hi for eu bs info anyway ??
            //correct the 2020 target - normalise total
            t.low=tlo*0.8f*fac - 100f; t.high=thi*0.7f*fac -100f; t.likely=(t.low+t.high)/2f;
            		//deb("targets for "+r+" in 2020: low: "+t.low+" likely: "+t.likely+" high: "+t.high);
            //make a 2030 target (with same relative shares as 2020, and -40% total - no low/high
            float t2030=tlo*0.6f*fac - 100f; 
            tt.add(new target(r, absreduc, allsec, K7gas, 1990, 2030, t2030, t2030, t2030));
            //and a 2050 target (with same relative shares as 2020), 80-95% reducn 
            float t2050low=tlo*0.2f*fac - 100f, t2050high=thi*0.05f*fac -100f, t2050likely=(t2050low+t2050high)/2f;
            tt.add(new target(r, absreduc, allsec, K7gas, 1990, 2050, t2050low, t2050high, t2050likely));
            endtarget.put(r, 2050);
        }
        fixedEU=true;
    }

    public void fillhistory() {
        for (region r : notarget) {
            for (int y = 1990; y <=hfos.ey; y++)   eq.emitfos_nat.set(r, y, hfos.get(r, y) / 1000f);
            for (int y = hfos.ey + 1; year <=2050; y++) eq.emitfos_nat.set(r, y, curve.dud);
        }
        for (int y = 1990; y <=  hfos.ey; y++) eq.emitfos_nat.set("bunker", y, eb.emitfosbase.get("bunker", y));
        //for (int y = 1990; y <=  hfos.ey; y++) deb("efb-bunker "+y+" "+eb.emitfosbase.get("bunker", y));
        for (int y = 1990; y <=  hfos.ey; y++) eq.emitfos_nat.calctot(y);
    }

} //end class
//========================



