/**
Infob is the base class for many objects in jcm that have basic information name, color, documentation, complexity etc.
including all interacobs (modules, params, qtsets) but also others (qts, regions, packages, doc-pages...)
This information is used especially for the tree, menu and documentation systems
As there are many instances of infob, data should be kept to a minimum
 */


package jcm.core.ob;

import java.util.*;
import java.lang.reflect.*;
import javax.swing.*;
import java.awt.Color;
import java.lang.annotation.*;
import jcm.core.*;
import jcm.core.cur.curveset;
import jcm.core.ob.*;
import jcm.core.par.*;
import jcm.core.cur.*;
import jcm.core.itf.*;
import jcm.core.tls.*;
import jcm.core.anno.*;
import jcm.gui.doc.*;
import jcm.gui.gen.*;
import jcm.gui.nav.*;
import jcm.script.calcscript;
import static jcm.core.complexity.*;
import static jcm.gui.gen.colfont.*;
import static jcm.gui.nav.filter.filtertype;
import static jcm.gui.nav.filter.filtertype.*;
import static jcm.core.anno.todo.type.*;
import static jcm.core.report.*;

public class infob implements menuFiller, Comparable<infob>, hasComplexity {

    //*** FIELDS *************************
    /**
    the name is a code used by labdoc for looking up labels and documentation, may also be used in scripting
    names are usually unique (but not necessarily - should be if preceded by owner name)
     */
    public String name = getClass().getSimpleName();
    /**  an infob's owner is another infob (typically a module) whose code contains its declaration and any special effects it may cause    */
    public infob owner = this;
    /**  the annotations associated with this infob  */
    public Annotation[] notes = null;
    /**  a set of "child" infobs, for use by the tree and menus  - only instantiated when filled, to save memory */
    public Set<infob> obs;
    /**  a set of actions for use by the tree and menus/popups  - only instantiated when filled, to save memory */
    public Map<jcmAction, filter.filtertype> actions;
    /** determines at which complexity level this infob is visible */
    public complexity mycomplexity = complexity.simplest;
    /** color for use in the tree etc. */
    public Color color = Color.black;
    /** higher priority infobs are displayed higher in trees/menus */
    public double priority = 1;
    /** if true, this infob is no longer active, should be ignored and eventually garbage-collected*/
    public boolean disposed = false;
    //********** working variables *************
    //maybe better to store these elsewhere if only used by one piece of code? 
    int level = 9999; //in tree, starting from rootob, 9999 implies not yet in tree
    double cump = 0;  //for tracking priority of added obs
    boolean expanded = false; //for use by tree

//******************************
    //CONSTRUCTOR
    /** a basic constructor to prevent subclasses calling infob varargs constructor by default*/
    public infob() {
        color = rcol();
    }

    /** varargs constructor  - can specify any of: name (String), owner (another infob), Color, priority, complexity */
    public infob(Object... args) {
        color = null;
        for (Object o : args) {
            if (o instanceof infob) {
                owner = (infob) o;
                if (color == null) color = (this instanceof world) ? rcol() : rcol(owner.color);
            }
            if (o instanceof String) {
                name = (String) o;
                checkinfo();
            }
            if (o instanceof Color) color = (Color) o;
            if (o instanceof Number) priority = ((Number) o).floatValue();
            if (o instanceof complexity) mycomplexity = (complexity) o;
        }
        if (color == null) color = Color.black;
    }

    //**************** DEFAULT COLORS / PRIORITIES **********
    //convenient to put here rather than in each module separately - but maybe even better to load a separate file or specify in package-info?
    //P3 colors / priorities for tree specified by text => not updating on refactoring!
    void checkinfo() {
        for (Object[] oo : packageob.setupinfo) if (removepackagename(name).equals((String) oo[0])) {
                priority = (Double) oo[1];
                if (oo.length == 3) color = (Color) oo[2];
                return;
            }
    }
    

    //**************************** ACTIONS *****************************
    public void addAction(jcmAction a) {
        addAction(null, a);
    }

    public void addAction(filter.filtertype f, jcmAction a) {
        if (actions == null) actions = new LinkedHashMap();
        actions.put(a, f);
    }

    /** add the default action for specified filtertype */
    public void addAction(filter.filtertype f) {
        if (actions != null && actions.containsValue(f)) return; //for unique actions, only make once
        if (f == f.Doc) addAction(f, docview.showdocaction(this));
        if (f == f.Tree) addAction(f, showpan.pan("Tree", jcmTree.class, "[" + this + "]"));
        if (f == f.interacmap) addAction(f, showpan.pan("interacmap", interacmap.class, this));
        if (f == f.Source) addAction(f, showpan.pan("Source", sourceview.class, getSourceFileName()));
        if (f == f.Data) addAction(f, showpan.pan("Data", sourceview.class, getDataFileName()));
    }

    //this converts the last / to a ., since root
    String getDataFileName() {
        String f = getName().replace(".", "/");
        int i = f.lastIndexOf("/");
        if (i > 0 && i < f.length()) f = f.substring(0, i) + "." + f.substring(i + 1, f.length());
        return f;
    }

    //winter08: changed source to src due to SVN 
    String getSourceFileName() {
        String s = ((this instanceof module) ? getClass().getName() : getName());
        return "src/" + s.replace(".", "/") + ".java";
    }

    void addClassActions() {
        try {
            final Class c = Class.forName(getName());
            if (java.lang.reflect.Modifier.isPublic(c.getModifiers())) addAction(filtertype.Source); //checking public is avoids finding inner classes for which no separate source file
            if (calcscript.class.isAssignableFrom(c)) addAction(filtertype.Scripts, new jcmAction(c.getSimpleName()) {

                    public void act() {
                        try {
                            ((calcscript) c.newInstance()).start();
                        } catch (Exception ex) {
                            deb(ex, "Problem Starting Script");
                        }
                    }
                });
        } catch (ClassNotFoundException e) {
            addAction(filtertype.Data);
        }
    }

    public boolean hasEnabledActions(Set<filtertype> fs) {
        if (actions != null) for (jcmAction a : actions.keySet()) if (checkenabled(a, fs)) return true;
        return false;
    }

    public filter.filtertype doFirstEnabledAction(Set<filtertype> fs) {
        if (actions != null) for (jcmAction a : actions.keySet()) if (checkenabled(a, fs)) {
                    a.act();
                    return actions.get(a);
                }
        return null;
    }

//********COMPLEXITY**********************
    public void mincomplexity() {
        mincomplexity(getObs());
    }

    public void mincomplexity(Collection<? extends infob> obs) { //sets complexity according to minimum of containing obs
        if (obs != null) {
            mycomplexity = experimental;
            for (infob o : obs) mycomplexity = o.getComplexity().compareTo(mycomplexity) < 0 ? o.getComplexity() : mycomplexity;
        }
    }

    public complexity getComplexity() {
        return mycomplexity != null ? mycomplexity : complexity.normal;
    }

    public boolean checkcomplexity() {
        return complexity.check(getComplexity());
    }

//************** ENABLED *******************
    public boolean checkenabled(Action a, filter.filtertype f) {
        return (actions.get(a) == null || actions.get(a) == f);
    }
//return f.toString().equals(a.getValue(a.NAME).toString());
    public boolean checkenabled(Action a, Set<filter.filtertype> fs) {
        for (filter.filtertype f : fs) if (checkenabled(a, f)) return true;
        return false;
    }

    public boolean checkenabled(filter.filtertype f) {
        return checkenabled(Collections.singleton(f));
    }

    public boolean checkenabled(Set<filter.filtertype> fs) {
        if (!checkcomplexity()) return false;
        if (this instanceof param && (fs.contains(AllParams) || (fs.contains(NeededParams) && ((param) this).checkneededforplot()))) return true;
        if (this instanceof curveset && fs.contains(Curves)) return true;
        if (actions != null) for (Action a : actions.keySet()) if (checkenabled(a, fs)) return true;
        if (obs != null) for (infob o : obs) if (o.checkenabled(fs)) return true;
        return false;
    }

//*************** FILL MENU *************
    public void fillMenu(jcmMenu m) {
        fillMenu(m, filter.all());
    }

    public void fillMenu(jcmMenu m, Set<filter.filtertype> fs) {
        if (actions != null) for (Action a : actions.keySet()) if (fs == null || checkenabled(a, fs)) m.add(a);
    }
    public static EnumSet singleaction = EnumSet.of(Doc, Scripts, Source, Data, Maps);

    public void fillTreeMenu(jcmMenu m) {
        List<infob> enabobs = getEnabledObs(m.filters);
        if (m.isTopLevelMenu() && enabobs != null && enabobs.size() == 1 && !hasEnabledActions(m.filters)) {
            enabobs.get(0).fillTreeMenu(m);
            return;
        } //if only one item in tree, skip to it
        fillMenu(m, m.filters);
        if (enabobs != null) for (infob i : enabobs) {
                if (m.filters.size() == 1 && singleaction.containsAll(m.filters) && i.getEnabledObs(m.filters) == null && i.actions != null) {
                    for (jcmAction a : i.actions.keySet()) if (i.actions.get(a) == m.filters.iterator().next()) m.add(a.clone(i.name));
                } else m.add(i);
            }
        if (m.filters.size() == 1) m.add(showpan.pan(docview.class, "About&" + m.filters.iterator().next() + "Menu"));
    }

//******** WORLD **********************
    public world getworld() {
        if (owner instanceof world) return (world) owner;
        if (owner instanceof module) return ((module) owner).world;
        if (owner == this) return null;
        return owner.getworld();
    }

    public module getmodule() {
        if (owner instanceof module) return (module) owner;
        if (owner == this) return null;
        return owner.getmodule();
    }

//********* OBS, OWNER  *******************
    public infob getOwner() {
        return owner;
    }

    public Set<infob> getObs() {
        return obs;
    }

    public List<infob> getEnabledObs(Set<filter.filtertype> fs) {
        if (fs.contains(filter.filtertype.Doc) && !expanded) expandDocTree();
        if (obs == null) return null;
        List<infob> eo = new ArrayList();
        for (infob o : obs) if (o != null && o.checkenabled(fs)) eo.add(o);
        if (eo.size() > 0) return eo;
        else return null;
    }

    public void addOb(infob b) {
        if (getObs() == null) obs = new TreeSet();
        getObs().add(b);
        b.level = level + 1;
        b.priority += cump;
        cump += 0.01; //cump increases, so we preserve the adding order (if no other criteria)
    }

    public int compareTo(infob b) {
        int i = (int) Math.signum(b.priority - priority);
        if (i == 0) i = name.compareTo(b.name);
        if (i == 0) return (this.equals(b)) ? 0 : this.hashCode() - b.hashCode(); //note important not to return zero unless they are really equal objects
        return i;
    }

//	   else return a.getClass().getName().compareTo(b.getClass().getName());
    public void removeOb(Object h) {
        if (getObs() != null) getObs().remove(h);
    }

    public infob find(String s) {
        return find(this, s);
    }

    public infob find(Object orig, String s) {
        String s2 = s.toLowerCase();
        if (obs != null) {
            for (infob o : obs) if (o.toString().equals(s)) return o;
            for (infob o : obs) if (o.toString().toLowerCase().equals(s2)) return o;
            for (infob o : obs) if (o != orig) {
                    infob oo = o.find(orig, s);
                    if (oo != null) return oo;
                } //recursive, beware loops!
        }
        return null;
    }

//******** DISPOSE ****************************
//clear all Obs and all children, beware recursive!
//note extended by interacob, but also called directly from world
    public void disposeLater() {
        //must run outside of loop - also maybe slow so let gui respond first
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                loop.waitUntilLoopDone();
                disposethis();
                jcmTree.restruclink.changed = true;
                loop.gonow();
            }
        });
    }

    public void disposethis() {
        if (disposed || ref.isstatic(this, owner)) return;   //avoid infinite loop in case circular owner links, and also ignore static iobs
        disposed = true;
        name += "&disposed";
        if (obs != null) {
            for (infob i : obs) i.disposethis();
            obs = null;
        }
    }

//***********COLOR ***************
    public Color getColor() {
        return color;
    }

    public String hashcolor() {
        return hashcolor(getColor());
    }   //color for html use in Jlabels
    //public static String hashcolor(Color c) {   return "<font color=#"+hex2(c.getRed())+hex2(c.getGreen())+hex2(c.getBlue())+" >";    }
    public static String hashcolor(Color c) {  //modified as Philippe suggestion,to make brightest colors darker
        int fac = (c.getRed() + c.getGreen() + c.getBlue()) < 512 ? 1 : 2;
        return "<font color=#" + hex2(c.getRed() / fac) + hex2(c.getGreen() / fac) + hex2(c.getBlue() / fac) + " >";
    }

    public static String hex2(int x) {
        return (x < 16 ? "0" : "") + Integer.toHexString(x);
    }

    public static Color rcol() {
        return new Color((int) (255 * Math.random()), (int) (255 * Math.random()), (int) (255 * Math.random())).darker();
    }

    public static Color rcol(Color b) {
        if (b == null) return rcol();
        else return new Color((int) (127f * Math.random() + b.getRed() / 2f), (int) (127f * Math.random() + b.getGreen() / 2f), (int) (127f * Math.random() + b.getBlue() / 2f)).darker();
    }

//************* NAME AND DOC ***************
    public String toString() {
        return getName();
    }

    public String getName() {
        return name;
    }

    public String getFullName() {
        return (owner != this ? owner.getFullName() + "&" : "") + name;
    }

    public String getNameWithWorld() {
        return world.worlds.size() > 1 ? (getworld() != null ? getworld().name + "&" : "null&") + name : name;
    }

    public static String removepackagename(String s) {
        int i = s.indexOf(".");
        return i > 0 ? removepackagename(s.substring(i + 1)) : s;
    }

    public String type() {
        return getClass().getSimpleName();
    } //for modules, same as name, but for params and qtsets different

    public String getLabel() {
        return labman.getShort(name);
    } //note for qt, switched to getTitle as the short form is very short (for legend)

    public String getTitle() {
        return labman.getTitle(name);
    }

    public String getExtraDoc() {
        return docNotes() + autodoc.javacodeforclass(name);
    }

    public String getSpecificDoc(Object... args) {
        return "<p><b>!specific doc request " + args[0] + " for " + name + " no longer implemented!</b><p> ";
    }

    public String docSummary() {
        try {
            getClass().forName(name);
            String n2 = removepackagename(name);
            return autodoc.link(n2) + "</font> " + "£%" + n2 + "<br>";
        } catch (ClassNotFoundException e) {
            return autodoc.link(this) + "</font> " + "£%" + labman.convertkey(name) + "<br>";
        }
    }

    public String docOwner() {
        return (owner != this) ? "<nobr>%%€€cogs £`memberof  " + autodoc.link(owner.name) + "%%</nobr><br>" : "";
    }

    public String docNotes() {
        String s = "";
        if (notes != null) for (Annotation a : notes) {
                s += "<hr>  == €€cogs  £` " + a.annotationType().getSimpleName() + " == ";
                if (a instanceof todos) for (todo t : ((todos) a).value()) s += docNote(t);
                else s += docNote(a);
            }
        return s;
    }

    public String docNote(Annotation a) {
        String s = "\n";
        try {
            for (Method m : a.annotationType().getMethods()) {
                String n = m.getName();
                if (n.equals("hashCode") || m.invoke(a).equals(m.getDefaultValue())) continue;
                Class c = m.getReturnType();
                s += "<font color=" + (c == String.class ? "blue" : "green") + ">" + (c == String.class || c.isEnum() ? " " : " <i>" + n + "</i>: ") + m.invoke(a) + "</font>, ";
            }
        } catch (Exception e) {
            deb(e);
        }
        return s;
    }

    //*********************************************
    /** used by getEnabledObs - to find all obs with hyperlinks from this (but avoiding recursion)  */
    void expandDocTree() {
        deb(name + " expand doc tree");
        List<String> links = new ArrayList(5);
        //get all links - a simplified copy of autodoc parse /sublab
        cs doc = new cs(labman.getDoc(name)); //+getExtraDoc()); - can't do this because calls getenabledobs =>expand doctree => ...
        doc.swap("@", "£@");
        StringBuilder p = doc.s;
        String type, result, openflag = "£", di, key;
        int i, j = 0, ofl = openflag.length();
        while ((i = p.indexOf(openflag, j)) >= 0) {
            j = doc.nextspace(i + ofl + 1);
            type = p.substring(i + ofl, i + ofl + 1);
            key = p.substring(i + ofl + 1, j);
            //catch link, subsection, linkplus, summary
            if (type.equals("@") || type.equals("£") || type.equals("*") || type.equals("%")) links.add(key);
        }

        addlink:
        for (String link : links) {
            deb("-" + link);
            label lab = labman.getLabel(link);
            if (lab == null) continue addlink;
            //P3 beware autodc.findob slow! maybe very inefficient - note also that constructor of label uses register.findob, but at the moment there is no other applciation of this jcmob
            if (lab.jcmob == null) lab.jcmob = autodoc.findob(link);
            if (lab.jcmob == null) {
                lab.jcmob = new infob(link, complexity.simplest);
                lab.jcmob.addAction(filter.filtertype.Doc);
            }
            infob ob = lab.jcmob;
            boolean onlydoc = ob.actions != null && ob.actions.keySet().size() == 1 && ob.actions.get(ob.actions.keySet().iterator().next()) == filter.filtertype.Doc;
            if (onlydoc && !(ob instanceof interacob) && !(ob instanceof packageob) && ob.level > level + 1) {
                if (ob.owner != null) ob.owner.removeOb(ob);
                addOb(ob);
                ob.owner = this;
                deb("-ADDED");
            }
        }
        expanded = true;
    }

//************ COMPONENT  & MENU & ICON ********************************
//maybe this should move to gui package?
    /** returns a JComponent illustrating this infob - overridden by params etc. */
    public JComponent getComponent(Object... args) {
        //used by treemaker, circmenu, qtset, param, overridden by param
        Box b = Box.createHorizontalBox();
        //new jcmMenu(b, this);
        JLabel jl = new JLabel(getLabel(), getIcon(), SwingConstants.LEFT);
        jl.setForeground(getColor());
        jl.setFont(this instanceof curveset ? colfont.italic : colfont.normalfont);
        b.add(jl);
        b.addMouseListener(showpan.moulist);
        b.addMouseMotionListener(showpan.moulist);
        showpan.moulist.cimap.put(b, this);
        return b;
    }

    /** returns associated Icon*/
    public Icon getIcon() {
        return iconFinder.findIcon(this);
    }
} //end class

