package jcm.core.cur;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import jcm.core.*;
import jcm.core.itf.hasColor;
import jcm.core.ob.*;
import jcm.core.par.param;
import jcm.gui.doc.autodoc;
import jcm.gui.doc.labman;

/**
one curve of data
<br>curve extends infob but not interacob (in order to avoid too many interactions and extra overhead), so all curves should be accessed via a curveset.
@see curve.Type
@see curveset
 */
public class curve extends infob {

    //******FIELDS***********************
    //note overlap with curveset - but they cn be different
    public int /** start year*/
            sy = module.gsy, /** end year*/
             ey = module.gey, /** timestep*/
             xstep = 1;
    /** flag indicating no valid data : note this is now set to NaN not -999 as in former versions but beware cannot use == with NaN !!! */
    public static final float dud = Float.NaN;

    /**
    There are several types of curve. This enum is also used by curveset.
    <br> Some curve types (normal, integral, total) actually store data in an array of floats,
    <br> other simple derivatives of other curves ( types difference, sum, ratio, rate and frac) just calculate on demand, to save memory
    <br>(these derivatives are mainly used for plots rather than inner-loop calculations)
     */
    public enum Type {

        /** just stores data in an array.*/
        normal,
        /** difference between two curves- calc on demand*/
        difference,
        /** sum of two curves- calc on demand*/
        sum,
        /** ratio of two curves- calc on demand*/
        ratio,
        /** rate of change of one curve, in % (dy/y.dt)- calc on demand*/
        ratefrac,
        /** rate of change of one curve, - calc on demand*/
        rate,
        /** fraction, in %, calc on demand */
        frac,
        /** indicates that this is a total, to be  excluded from "map without total" (used for some calculations and stacked plots) - stores data */
        total,
        /** integral: stores data, since not efficient to recalc from beginning for each year*/
        integral,
        /** used when adding third curvesets for scatterplots etc. */
        smooth,
        /** applies a moving average, to smooth out short term variations */
        extra
    }
    /**type of this curve*/
    public Type type = Type.normal;
    //main array to store data, this is private, so could change the internal data storage structure (e.g. variable timestep)
    private float[] a;
    //used for derivatives
    private curve qa,  qb;
    /** indicates line style */
    public enum linestyle { line, dotted, dashed, dotdash }
    public linestyle style=linestyle.line;

    //********CONSTRUCTOR*********************
    /**
    curve constructor can take a variety of arguments<ul>
    <li>a String = it's name (for labels etc.)
    <li>a Color (for plots)
    <li>a complexity level
    <li>a curve.Type
    <li>integers to set sy, ey, xstep
    <li>an interacob that is its owner
    <li>an array of floats already containing data
    <li>one or two other curves: used to make derivatives - also setting the name, color, and timescale
    <li>boolean true = clone (copy) the  first curve
    </ul>
     */
    public curve(Object... args) {
        int ic = 0;
        mycomplexity = complexity.normal;
        boolean clone = false;
        color = rcol();

        argloop:
        for (Object o : args) {
            if (o instanceof curve) {
                if (qa == null) {
                    qa = (curve) o;
                    color = qa.color;
                    name = qa.name;
                    sy = qa.sy;
                    ey = qa.ey;
                    xstep = qa.xstep;
                } else qb = (curve) o;
                continue argloop;
            }
            if (o instanceof Boolean) {
                clone = (Boolean) o;
            }
            if (o instanceof interacob) {
                owner = (interacob) o;
            }
            if (o instanceof float[]) {
                a = (float[]) o;
            }
            if (o instanceof hasColor) {
                color= ((hasColor)o).getColor();
            }
            if (o instanceof Color) {
                color = (Color) o;
            }
            if (o instanceof complexity) {
                mycomplexity = (complexity) o;
            }
            if (o instanceof String) {
                name = (String) o;
            }
            if (o instanceof Number) {
                int i = ((Number) o).intValue();
                switch (ic) {
                    case 0:
                        sy = i;
                        break;
                    case 1:
                        ey = i;
                        break;
                    case 2:
                        xstep = i;
                        break;
                }
                ic++;
            }
            if (o instanceof Type) type = (Type) o;
        } //args

        if (a == null && (clone || type == Type.normal || type == Type.total || type == type.integral)) a = new float[1 + (ey - sy) / xstep];
        if (type == Type.integral) calcintegral();

        if (clone) {
            if ((type == Type.normal || type == Type.total) && qa.xstep == xstep) try {
                System.arraycopy(qa.a, (sy - qa.sy) / xstep, a, 0, a.length);
            } catch (Exception e) { System.out.println("CLONE PROBLEM INFO: "+qa.a+" "+a+" "+(sy - qa.sy) / xstep+" 0 "+a.length); e.printStackTrace(); }
            else {
                for (int y = sy; y <= ey; y += xstep) set(y, get(y));
                type = Type.normal; qa=null; qb=null;
            }
        }
    }

    //******** CLONE *********************
    /** 
     This copies the curve, and it's data, via the clone method in constructor (above)
     Resulting type is always normal, links to qa,qb or owner are not copied, thus it is independent of the original.
     */
    public curve cloneIndependent(Object... args) {
        Object[] ea;
        if (type==Type.ratio)  ea=new Object[]{true, qa, qb, Type.ratio};
        else ea=new Object[]{true, this};
        Object[] args2 = new Object[args.length + ea.length];
        System.arraycopy(ea, 0, args2, 0, ea.length);
        System.arraycopy(args, 0, args2, ea.length, args.length);
        return new curve(args2);  
    }

    //********** CHANGE YEAR ************
    /** changes the internal array structure - slow! */
    public void changeendyear(int oldey, int newey) {
        if (ey == oldey) {
            if (a != null) { //copy, slow!
                float[] b = new float[1 + (newey - sy) / xstep];
                System.arraycopy(a, 0, b, 0, Math.min(a.length, b.length));
                a = b;
            }
            ey = newey;
        }
        if (qa != null) qa.changeendyear(oldey, newey);
        if (qb != null) qb.changeendyear(oldey, newey); //maybe unecessary if already found via another curveset?
    }

    //****** GET,  SET ***********************
    public static boolean ok(float f) {
        return !Float.isNaN(f);
    }

    /** true if year within timescale and get(year) is not curve.dud, NaN or infinite*/
    public boolean gotdata(int year) {
        return year >= sy && year <= ey && !Float.isNaN(get(year)) && !Float.isNaN(get(year)) && !Float.isInfinite(get(year));
    }

    /** get the value for the current module.year*/
    public float get() {
        return get(module.year);
    }

    /** get the value for the specified year:
    <br> for normal, total or integral curves this returns the stored data
    <br> for other derivative curves it calculates the data
    <br> note: in case of missing data or division by zero etc., will return curve.dud
     */
    enum smoothopt { even, triangular }; 
    public static param<smoothopt> smoothtype=new  param("smooth type", smoothopt.values(), smoothopt.triangular);
    public static param smoothinterval=new  param("smooth spread", "", 4, 0, 7);
    
    public float get(int year) {
        switch (type) {
            case difference: {
                return (qa != null && qb != null && ok(qa.get(year)) && ok(qb.get(year))) ? qa.get(year) - qb.get(year) : dud;
            }
            case sum: {
                return (qa != null && qb != null && ok(qa.get(year)) && ok(qb.get(year))) ? qa.get(year) + qb.get(year) : dud;
            }
            case ratio: {
                return (qa != null && qb != null && ok(qa.get(year)) && ok(qb.get(year)) && qb.get(year) != 0) ? qa.get(year) / qb.get(year) : dud;
            }
            case frac: {
                return ok(qa.get(year)) ? 100 * qa.get(year) / ((curveset) owner).calctot(year) : dud;
            }
            case rate: {
                return (ok(qa.get(year)) && ok(qa.get(year - xstep))) ? (qa.get(year) - qa.get(year - xstep)) / xstep : dud;
            }
            case ratefrac: {
                return (ok(qa.get(year)) && ok(qa.get(year - xstep)) && qa.get(year - xstep) != 0) ? (100f * qa.get(year) / qa.get(year - xstep) - 100f) / xstep : dud;
            }
            /** inefficient!  - temporary fix - for moving average should just add new one and remove old*/
            case smooth: {
                float result = dud, next; int num=0, q=(int)smoothinterval.getval(), w=1;
                for (int i=-q; i<=q; i++) { 
                    if (smoothtype.chosen==smoothopt.triangular) w=(int)smoothinterval.getval()+1 -Math.abs(i);
                    next=qa.get(year+i);  if (ok(next)) { num+=w; result= (ok(result) ?   result+next*w : next*w); }
                } 
                return  (num>0 ? result/num : dud);
            }
            case integral:
            case total: //same as below
            default:
                if (year >= sy && year <= ey) {
                    if (xstep == 1) {
                        return (ok(a[year - sy])) ? a[year - sy] : dud;
                    } //quick
                    int n = (year - sy) / xstep, r = (year - sy) % xstep;
                    float f = (r == 0) ? a[n] : a[n] + (a[n + 1] - a[n]) * ((float) r / xstep); //interpolates linearly
                    if (ok(f)) return f;
                }
                return dud;
        }
    }

    /** set value for the current module.year */
    public void set(float f) {
        set(module.year, f);
    }

    /** set value for specified year (<i>use only for normal type curves, not derivatives curves!</i>)*/
    public void set(int year, float f) {
        if (year >= sy && year <= ey) a[(year - sy) / xstep] = f;
    }

    //******** internal data ***************
    /** the internal array of float data (maybe null for derivatives) */
    public float[] geta() {
        return a;
    }

    /** the year corresponding to the nth data item (used for table) */
    public int getyear(int n) {
        return sy + xstep * n;
    }

    /** the length of the array of data, or (for derivatives) that of  parent curve*/
    public int getlength() {
        return type == Type.normal || type == Type.total ? a.length : qa.a.length;
    }

    //********* integral ***********
    //note called from both constructor and curveset postcalc
    void calcintegral() {
        /* 
        P2 disabled (quick fix for MATCH) to calc integral only between start and end year  - should have own param 
        responsibility resp=world.worlds.get(0).gm(responsibility.class);
        int isy=(int)resp.startyear.getval(), iey=(int)resp.endyear.getval();
         */
        float tot = 0;
        int isy = sy, iey = ey;
        for (int y = isy; y <= iey; y += xstep) {
            //	for (int y=sy; y<=ey; y+=xstep ) {
            tot += (ok(qa.get(y)) ? qa.get(y) : 0); //note ignores dud
            a[(y - sy) / xstep] = tot;
        }

    }

    //********************* MAX /MIN ************
    /** maximum value over time-range */
    public float getmax() {
        float max = Float.MIN_VALUE;
        for (int y = sy; y < ey; y += xstep) if (ok(get(y)) && get(y) > max) max = get(y);
        return max;
    }

    /** minimum value over time-range */
    public float getmin() {
        float min = Float.MAX_VALUE;
        for (int y = sy; y < ey; y += xstep) if (ok(get(y)) && get(y) < min) min = get(y);
        return min;
    }

//****************** DOC / INFO ************
    /** get label for plot: uses medium (title) form */
    public String getLabel() {
        return labman.getTitle(name);
    }

    /** wiki-doc summary */
    public String docSummary() {
        String name2 = labman.convertkey(name);
        return hashcolor() + autodoc.link(name2, getTitle() + labman.getShortIfDifferent(name2)) + "</font> " + "£%" + name2 + "<br>";
    }
//" <br>%%"+ getIcon()+ labman.getDoc(name)+"%%<br>"; } //problem is that doesn't stop further parsing
    /** wiki-doc about owener and coe-link*/
    public String getExtraDoc() {
        return docNotes() + docOwner() + autodoc.javacode(owner);
    }
} //end class






