package jcm.mod.cli;

import java.util.EnumSet;
import jcm.core.*;
import jama.*; //this is the java matrix package for calculating eigenvectors, inverses etc.
import java.awt.Color;
import java.util.EnumMap;
import jcm.core.anno.*;
import jcm.core.par.param;
import jcm.core.cur.curve;
import jcm.core.ob.module;
import static jcm.gui.gen.colfont.*;
import static jcm.core.complexity.*;
import static jcm.core.report.*;
import static jcm.mod.cli.udebclimod.cp.*;
import static jcm.mod.cli.udebclimod.gcm.*;
import static jcm.mod.cli.udebclimod.gcmset.*;


/**
 Climate model to calculate global temperature rise from radiative forcing
 Based on Wigley-Raper Upwelling-Diffusion Energy Balance model
 fitted to GCMs according to IPCC AR4 or TAR
 and solved using eigenvector method
 also calculates thermal expansion due to sea-level rise
 
 This module contains the main calculation routines of the UDEB model
 These routines are called from glotemp,
 but separated from glotemp so that the eigenvectors are only recalculated (slow!) when the model setup changes
 (also to make the code more managable, and to create potential structure for swapping with other climate models)
 
 Developed in Copenhagen spring 2001, updated Louvain la Neuve spring 2007
 
  */

public class udebclimod extends module  {
    
    
    
    //*****************************************************
    //INTERACTIONS
    
    public void initsetup() {
        makeparams();
        gp.get(rfco2double).setaffects(gm(radfor.class));
    }
    
    public void setinteractions() {
        setaffectedby(gp.get(zeroupwelltemp), varupwell.istrue());
        setaffectedby(gp.get(upwellreducefrac), varupwell.istrue());
        setaffectedby(gp.get(upwellbaserate), varupwell.istrue());
    }
    
    //****************** tuning to GCMs *********************
    
    public enum cp {
        
        rfco2double("rfco2d", "w&per&m2", 3.71, 3, 5,  black, normal),
        climsens("climsens", "degc", 3.0, 1.5, 7.5, red, simplest),
        climsens_gcm("cs_gcm", "degc", 3.21, 1.5, 7.5, red, expert),
        climsens_effective("cs_effective", "degc", 2.925, 1.5, 7.5, dkred, expert), //see explanation IPCC-AR4 section 8.8.2
        transientclimateresponse("tcr", "degc", 1.8, 1, 3, purple, experimental),
        heatdiffusivity("teddydiff", "cm2&per&s", 1.9, 0.05, 4, dkblue, normal),
        landoceantempratio("lotr", "", 1.4, 0.7, 2.1, orange, expert),
        zeroupwelltemp("tnoupwell", "degc", 8, 0, 30, lilac, expert),
        knorthsouth("kns", "",  1, 0, 4, dkgreen, expert),
        klandocean("klo", "", 1, 0, 2, yellowgreen, expert),
        mixlaydepth("tmixlay", "metres", 60, 1, 120, grey, expert),
        polarsinktempratio("psi", "",  0.2, 0, 1, blue, expert),
        seaice("seaice", "", 1.25, 0.8, 1.6, cyan, expert),
        upwellreducefrac("uwredfrac", "", 0.3, 0, 1, dkpurple, expert),
        upwellbaserate("uwbaserate", "m&per&yr", 4, 0, 8, pink, expert);
        
        //info for param, but note we can't put the param in the enum, because would be static
        //note param defaults are set to the default AR4 model, these defaults are used where no data for a specific GCM
        String lab, units; double def, min, max; Color co; complexity cx;
        cp(String l, String u, double d, double a, double b, Color c, complexity x) { lab=l; units=u; def=d; min=a; max=b; co=c; cx=x; }
    }
    
    static cp[]
            ar4p =  {rfco2double,  climsens_effective, climsens_gcm,   transientclimateresponse, heatdiffusivity, landoceantempratio},
            tarp=    { rfco2double, climsens, heatdiffusivity, landoceantempratio, zeroupwelltemp, knorthsouth, klandocean },
            sarp= { rfco2double, climsens, heatdiffusivity, landoceantempratio, zeroupwelltemp, knorthsouth, klandocean, mixlaydepth, polarsinktempratio, seaice};
    
    
    public enum gcm {
        
        //IPCC AR4 GCMs - see FOD supplementary material table 8.SM1, text section 8.8, table 8.2
        AR4_average	 (3.642,  2.925,   3.211, 1.763,  1.899,   1.391),  //Calculated from data below
        
        CCS_M3                 (3.95,   2.37,  2.7, 1.5,  1.73,   1.18),
        CGC_M31_T47    (3.32,   3.02,   3.4, 1.9,  1.57,   1.58),
        CNRM_CM3          (3.71,   2.45,  2.45, 1.6,  1.21,   1.10),  //no value for CS given => use effective
        CSIRO_Mk3          (3.47,   2.21,   3.1, 1.4, 2.03,   1.33),
        ECHAM5_MPIOM(4.01,   3.86,   3.4, 2.2, 1.22,   1.41),
        ECHO_G                (3.71,   3.01,   3.2, 1.7, 2.01,   1.65),
        FGOALS_g1          (3.71,   1.97,   2.3, 1.2, 4.57,   1.64),
        GFDL_CM2           (3.50,    2.35,   2.9, 1.6,   1.42,    1.47),
        GFDL_CM2_1      (3.50,    2.28,    3.4, 1.5, 2.23,    1.58),
        GISS_EH              (4.06,    3.04,   2.7, 1.6,  2.35,    1.21),
        GISS_ER               (4.06,    2.57,   2.7, 1.5,  4.42,    1.44),
        INM_CM3              (3.71,    2.28,   2.1, 1.6,  0.79,    1.10),
        IPSL_CM4             (3.48,    3.83,   4.4, 2.1,  1.94,    1.26),
        MIROC3_2_HI      (3.14,    5.87,   4.3, 2.6,  1.18,    1.15),
        MIROC3_2_MED(3.09,    3.93,   4.0, 2.1,  2.29,    1.58),
        MRI_CGCM_232(3.47,    2.97,   3.2, 2.2,  1.22,    1.45),
        PCM		        (3.71,    1.88,   2.1, 1.3,  1.57,    1.45),
        HadCM3             (3.81,    3.06,    3.3, 2.0, 1.01,    1.65),
        HadGEM1           (3.78,    2.63,   4.4, 1.9,  1.32,    1.20),
        
        //IPCC TAR GCMs follow - see IPCC tarp WG1 Appx 9.1
        GFDL_TAR	 (tarp, 3.71,  4.2,   2.3,   1.2,   8.0,    1.0, 1.0),
        CSIRO_TAR       (tarp, 3.45,  3.7,   1.6,   1.2,   5.0,    1.0, 1.0),
        HadCM3_TAR  (tarp, 3.74,  3.0,   1.9,   1.4,   25.0,  0.5, 0.5),	    //Note  - use 3.691 for RF for CO2 double for consistency with ACCC - HADCM3
        HadCM2             (tarp, 3.47,  2.5,   1.7,   1.4,   12.0,  0.5, 0.5),
        ECHAM4             (tarp, 3.8,     2.6,   9.0,   1.4,   20.0,  0.5, 0.5),	    //Note strange value for diffusivity - but this is consistent with tarp
        CSM                     (tarp, 3.6,     1.9,   2.3,   1.4,   9999.0, 0.5, 0.5),	     // -9999 for no upwelling feedback
        DOE                     (tarp, 3.6,     1.7,   2.3,   1.4,   14.0,  0.5, 0.5),
        
        //IPCC SAR- note deeper mixed layer and seaice=1 (none)
        IPCCSAR            (sarp, 4.37,   2.5,   1.0,   1.3,   7.0,    1.0, 1.0, 90, 0.2, 1.0),
        ;
        
        //Note: RF CO2double HadCM3=3.74 =>/log2 => 5.396 but ACCC has 5.325 (RF2x=3.691, 1.33% less)  and TAR says 5.35
        
        EnumMap<cp, Double> val=new EnumMap(cp.class); // this stores the data
        
        //Constructor
        gcm(double ...x) { this(ar4p, x); }
        gcm(cp[] pars, double ...x) {  int i=0; for (cp p : pars) { val.put(p, x[i]); i++; } }
    }
    
    
    
    enum gcmset {
        AR4(EnumSet.range(AR4_average, HadGEM1)),
        AR4_selection( EnumSet.of(HadGEM1, MIROC3_2_MED, ECHAM5_MPIOM, HadCM3, CSIRO_Mk3, GISS_EH)),
        TAR(EnumSet.range(GFDL_TAR, DOE)),
        SAR(EnumSet.of(IPCCSAR)),
        all_GCMs(EnumSet.allOf(gcm.class))
        ;
        
        EnumSet<gcm> gcms;
        gcmset(EnumSet<gcm> eg) { gcms=eg;  }
        gcm[] gcmlist() { return gcms.toArray(new gcm[gcms.size()]); }
    }
    
    
    //*****************************************************
    //PARAMETERS
    //adjustable parameters -may be changed by controls on the graphs
    
    EnumMap<cp, param> gp =new EnumMap(cp.class); //map of all the params set by gcms
    public double  val(cp p) { return gp.get(p).getval(); } //shortcut
    
    
    public param<gcmset> climodtype=new param("climodtype", gcmset.values(), gcmset.all_GCMs ) {
        public void precalc() {     climod.setlist( ((gcmset)chosen).gcmlist()) ; }
        //note effect of this: - if chosen GCM is not in new list, will appear blank, but no parameters change
    };
    
    public param<gcm>  climod=new param("climodmenu", climodtype.chosen.gcmlist(), AR4_average, simplest) {
        public void precalc() {
            gcm g=(gcm)chosen;
            for (cp p : cp.values()) if (p!=cp.climsens) { gp.get(p).putval( g.val.containsKey(p) ? g.val.get(p) :  p.def  ); }
            if (csopt.chosen==csopts.user_cs);  // do nothing - keep value of climsens param  
            if (csopt.chosen==csopts.use_effective_cs && g.val.containsKey(cp.climsens_effective)) gp.get(cp.climsens).putval(g.val.get(cp.climsens_effective));
            if (csopt.chosen==csopts.use_gcm_cs && g.val.containsKey(cp.climsens_gcm)) gp.get(cp.climsens).putval(g.val.get(cp.climsens_gcm));
            setupfluxes();
            gm(sealevel.class).seticecaptomod((gcm)chosen);
        }
        //note need to put this precalc in the param not the module, so it will still work even if module output flag false
        //beware maybe called after main precalc ?? (old note)
    };
    
    void test() {
        udebclimod ud=gm(udebclimod.class);
    }
    
    public param  varupwell=new param("tufbopt", true, magenta, expert);
    enum csopts     { use_effective_cs, use_gcm_cs, user_cs  }
    
    public  param<csopts> csopt=new param("cs_source", csopts.values(), csopts.user_cs, dkred, expert) { public void precalc() { climod.precalc(); }};
    
    
    void makeparams() {
        gcm g=climod.chosen;
        for (cp c: cp.values()) {
            param p=new  param(c.lab, c.units, g.val.containsKey(c) ? g.val.get(c) : c.def, c.min, c.max, c.co, c.cx );
            gp.put(c, p);
            p.priority-=0.2; //push down after the others
            allparam.add(p); p.setaffects(this); p.owner=this;  addOb(p); //methods normally handled by reflection in module, won't find inside EnumMap
            p.setaffectedby(climod);
        }
    }
    
    
    public void precalc() {
        if (!climod.changed) setupfluxes(); //if ! to avoid doing it twice
    }
    
    
    
    //*********** UDEB Model starts below *****************************************
    //working variables
    //[2] is for two oceans, 0=N, 1=S (although they only differ at surface)
    
    //nhl =num ocean layers (excluding surface), nhb=total num boxes
    //can try varying this - increasing it doesn't make much difference
    //note Jan07 nhl increased from 34 to 39 to be consistent with IPCC AR4 WG1 FOD 8.8.2
    public static final int nhl=39, nhb=nhl+1;
    
    double[][] hbox(){	return new double[2][nhb]; }
    
    //hiq stores data, hiqi for iteration, hrML for querying mixed layer, hicML column for input , premultiplied by stepf and rampf
    public double[][] hiq=hbox(), hiqi=hbox(), hiq99=hbox(), hpropf=hbox(), hrML=hbox(),  shicML=hbox(), rhicML=hbox();
    //for sea level rise
    public double[][] hrsl=hbox();
    
    //*****************************************************
    //for surface temperatures
    //area fractions, nl, no, so, sl (also needed by radfor module)
    //note 0=nl, 1=no, 2=so, 3=sl;
    public double[] frac={	0.199, 0.301, 0.406, 0.094};
    //swap with below for equal n-s areas
    //{0.293/2.0, 0.707/2.0, 0.293/2., 0.707/2.0};
    double fracl=frac[0]+frac[3], fraco=1.0-fracl;
    double[][] khi=new double[4][4]; //inverse of conductivity matrix * frac
    double[] khir=new double[4]; //sum of each row of above
    
    public double tstart=18.17; // average mix layer temp consistent with hilda
    
    //*****************************************************
    //for non-linear fluxes etc.
    public double kls, kos, klo, kns, nstd, nstdold, dqin, qpt, sl, cice;
    double[] qin=new double[2], qinbase=new double[2], qinold=new double[2], qinold99=new double[2], mlt=new double[2];
    public double[] spaceflux=new double[2];
    double totrf99, nstd99, nstdold99;
    
    //****************************************************
    //various working variables
    double dt, hu, pi, diffu, dml, rvu, rv, rjff, polsink, mlbaserate, hiquc, reftemp=0;
    double[] dl, vec;
    double[][] rate;
    public double[][] hiqstart=new double[2][nhb];
    
    Matrix M, U; Matrix[] MV=new Matrix[2], MVI=new Matrix[2];
    double upwellstep[][][] =new double[2][nhb][nhb];
    int startyear, tstep;
    double guess, nit, ciqr, diff;
    
    //**********************************************
    //heat flux system setup
    //note units: rates per year, ocean boxes contain watt-years per m2, distances in metres
    //note ocean boxes contain heat (per m2), not temperature, for rate formulae same as carbon model,
    
    public void setupfluxes() {
        
        //**********************
        //setup various constants
        dt=1.0;
        dml=val(mixlaydepth); // mixed layer depth 90 is IPCCSAR, 60 for IPCCTAR, note hilda (carboncycle)is 75!
        dl=new double[nhl]; // depth of ocean layers
        
        for (int n=0; n<nhl; n++) {	dl[n]=(n<20 ? 49.0 : 196.0)*(34.0/nhl); } //two thicknesses as hilda
        //		for (n=0; n<nhl; n++) {	dl[n]=3724/nhl; } //for evenly spaced layers
        
        diffu=val(heatdiffusivity)*(365*24*3600)/10000; // diffusivity -fixed, converted from cm2 s-2 to m2 yr-1
        
        hu=val(upwellbaserate); // upwelling m yr-1 -fixed for now, but note >> hilda!
        pi=val(polarsinktempratio); // PI factor for calculating temperature of high-latitude downwelling water
        
        double cw=4100000.0/(24.0*365.0*3600.0); // 0.13, watt-year per m3 ocean per K -function of heat capacity
        qpt=cw*dml; //heat uptake by mixed layer watt-year per m2 per K
        
        klo=val(klandocean); kns=val(knorthsouth); // land-ocean and north-south conductivities
        double lotr=val(landoceantempratio); // eqm land-ocean temperature ratio
        cice=val(seaice); // sea-ice parameter (see raper et al 2000)
        
        //set area fractions moved to reset method cos used by radfor setup which is called before climate setup!
        
        
        //*********************************
        //iteration to find conductivities for ocean-space and land-space (kos and kls)
        //checked works with my mathcad file, takes about 7 loops
        
        //kos=2.26589, kls=0.78311;	consistent with fracs and lo=0.5, ns=0.5, climsens=2.5, lotr=1.3
        
        double clotr, oldclotr=1.0, newkos, oldkos=-99.0; //-99 is just a flag for first loop
        kos=0.9; //first guess
        
        do{
            
            kls=((val(rfco2double)/val(climsens))*(lotr*fracl + fraco)/(lotr*fracl))-((kos*fraco)/(lotr*fracl));
            
            khi= new Matrix(new double[][]{
                {	frac[0]*kls+klo, -klo, 0, 0},
                {	-klo, frac[1]*kos+klo+kns, -kns, 0},
                {	0, -kns, frac[2]*kos+klo+kns, -klo},
                {	0, 0, -klo, frac[3]*kls+klo}
            },4,4).inverse().getArray();
            
            //multiply khi by the area fracs, and sum to calculate temp/rf
            for (int i=0; i<4; i++){	khir[i]=0; for (int j=0; j<4; j++) {	khi[i][j]*=frac[j]; khir[i]+=khi[i][j]; }}
            //calc land-ocean temp ratio
            clotr=((frac[0]*khir[0]+frac[3]*khir[3])/fracl)/((frac[1]*khir[1]+frac[2]*khir[2])/fraco);
            //adjust kos
            if (oldkos==-99.0) newkos=1.0;
            else newkos=kos+(lotr-clotr)*(kos-oldkos)/(clotr-oldclotr);
            oldkos=kos; kos=newkos; oldclotr=clotr;
            
        } while (Math.abs(lotr-clotr)>0.0001);
        
        //deb("kos "+kos+" kls "+kls);
        
        //*********************************
        //make rates for exchange between boxes (gross rates per year)
        
        //boxes 0 to nhl-1 are the layers below mixed layer, nhl is the surface (note nhb=nhl+1)
        //this numbering is for consistency with carbon setup
        //rate[i][j] is the amount by which you have multiply box j to get the change in box i
        
        
        rate=new double[nhb][nhb]; for (int i=0; i<nhb; i++) {	for (int j=0; j<nhb; j++) {	rate[i][j]=0; }}
        
        //go through layers
        for (int n=1; n<nhl; n++) {
            rvu=(2.0*diffu-hu*dl[n])/(dl[n-1]*(dl[n-1]+dl[n]));
            rv =(2.0*diffu+hu*dl[n-1])/(dl[n]*(dl[n-1]+dl[n]));
            rate[n][n-1]+=rvu; rate[n-1][n-1]-=rvu;
            rate[n-1][n]+=rv; rate[n][n]-=rv;
        }
        
        //top layer 0 and mixed layer nhl
        rvu=(8.0*diffu/(3.0*dl[0]*dml))-hu/dml;
        rv =3.0*diffu/(dl[0]*dl[0]);
        rjff=diffu/(3.0*dl[0]*dl[0]); //joos funny flux -see carbon model
        rate[0][nhl]+=rvu; rate[nhl][nhl]-=rvu;
        rate[nhl][0]+=rv; rate[0][0]-=rv;
        rate[0][1]+=rjff; rate[nhl][1]-=rjff;
        
        //polar water sinking to bottom
        polsink=pi*hu/dml;
        rate[nhl-1][nhl]+=polsink; rate[nhl][nhl]-=polsink;
        
        //record this before adding heat flux to mixed layer later
        mlbaserate=rate[nhl][nhl];
        
        M=new Matrix(rate,nhb,nhb);
        
        //*******************************
        if (varupwell.istrue()) {
            //calculate function for changing upwelling
            //matrix stepf/zut*(S-1.U.S) for *(S-1.Q) where zut is zeroupwelltemp
            
            double[][] urate=new double [nhb][nhb];
            for (int n=0; n<nhb; n++) for (int m=0; m<nhb; m++) urate[n][m]=0;
            
            //go through layers
            for (int n=1; n<nhl; n++) {
                rvu=(-hu*dl[n])/(dl[n-1]*(dl[n-1]+dl[n]));
                rv =(hu*dl[n-1])/(dl[n]*(dl[n-1]+dl[n]));
                urate[n][n-1]+=rvu; urate[n-1][n-1]-=rvu;
                urate[n-1][n]+=rv; urate[n][n]-=rv;
            }
            
            //top layer 0 and mixed layer nhl
            rvu=-hu/dml;
            urate[0][nhl]+=rvu; urate[nhl][nhl]-=rvu;
            
            //polar water sinking to bottom
            polsink=pi*hu/dml;
            urate[nhl-1][nhl]+=polsink; urate[nhl][nhl]-=polsink;
            
            U=new Matrix(urate,nhb,nhb);
        } //end if varupwell
        
        //*******************************
        //calculate starting mixed later heat contents -assuming same for both oceans
        double mlstart=tstart*qpt;
        
        //find initial box contents
        //assume all net fluxes zero except for surface layer
        //use inverse of submatrix excluding mixed layer (rest same for both oceans)
        //since M(nhl,nhl) excluded, doesn't matter it's not set yet!
        Matrix veclay=M.getMatrix(0,nhl-1,0,nhl-1).inverse().times(M.getMatrix(0,nhl-1,nhl,nhl)).times(-mlstart);
        double[] hss=veclay.transpose().getArray()[0];
        
        //the section below -up to vec is for sea-level rise
        //note this method assumes volume expansion coeff constant within one box (i.e. based on initial t,p)
        
        //convert to temperatures
        double[] tss=new double[nhb]; tss[nhl]=tstart;
        for (int n=0; n<nhl; n++) tss[n]=hss[n]/(cw*dl[n]);
        //to check initial temps ok //for (n=0;n<nhl;n++) System.out.println(""+n+" "+tss[n]);
        
        //pressure: note pr[0]= depth of first layer
        double pr[]=new double[nhb]; double pz= 9.8 * 1035.0 / 2.0; //should really vary!
        pr[nhl]=101325.0+pz*dml; pr[0]=pr[nhl]+pz*(dml+dl[0]);
        for (int n=1; n<nhl; n++) pr[n]=pr[n-1]+pz*(dl[n-1]+dl[n]);
        
        //calculate volume expansion coefficient -this approximate formula found by fitting to curves in mathcad file
        //note divide by cw so it can be multiplied by watt-yr per m2, the layer depth will cancel
        vec=new double[nhb];
        for (int n=0; n<nhb; n++) vec[n]=(1.0/cw)*(1.0e-4*(0.5 + 0.12*tss[n] - 0.0008*tss[n]*tss[n] + 2.8*(1.0 - 0.027*tss[n])*(pr[n]*1.0e-8 - 0.24*pr[n]*pr[n]*1.0e-16)));
        
        //check for (n=0;n<nhb;n++) modloop.debug.write(n+" "+tss[n]+" "+pr[n]+" "+vec[n]);
        
        //*******************************
        //work out prop, ramp, step functions, including heat loss to space
        
        double[] Eig=new double[nhb], stepf=new double[nhb], rampf=new double[nhb], hicML=new double[nhb];
        
        for (int  o=0; o<2; o++) {	//for each ocean -different surface losses considering neighbouring land!
            
            //flux lost to space from mixed layer -note still per m2, o*3 gives land box (N or S)!
            spaceflux[o]=(1/qpt)*cice*(kos + (klo/frac[o+1])*(kls*frac[o*3]/(kls*frac[o*3]+klo)));
            M.set(nhl,nhl,mlbaserate - spaceflux[o]);
            
            // now put this into a jama Matrix R
            EigenvalueDecomposition ED=new EigenvalueDecomposition(M);
            MV[o]=ED.getV(); MVI[o]=MV[o].inverse();
            
            //set hiqstart for initial state in 1750
            for (int n=0; n<nhb; n++) {
                hiqstart[o][n]=0;
                for (int m=0; m<nhb; m++) hiqstart[o][n]+=(MVI[o].getArray()[n][m])*(m<(nhb-1) ? hss[m] : mlstart);
            }
            
            //make hpropf, hstepf, hrampf functions
            Eig= ED.getRealEigenvalues();
            for (int n=0; n<nhb; n++) {
                hpropf[o][n]=Math.exp(Eig[n]*dt);
                if (Math.abs(Eig[n])<0.000001) {	stepf[n]=dt; rampf[n]=dt/2.0; } else {	stepf[n]=(hpropf[o][n]-1.0)/Eig[n]; rampf[n]=(stepf[n]-dt)/(Eig[n]*dt); }
            }
            
            //also need the row of S for the mixedlayer box, and column of S-1 premultiplied by step or ramp
            hrML[o]=MV[o].getArray()[nhl];
            hicML=MVI[o].transpose().getArray()[nhl];
            for (int n=0; n<nhb; n++) {	rhicML[o][n]=rampf[n]*hicML[n]; shicML[o][n]=stepf[n]*hicML[n]; }
            
            //also make a vector for calculating total sea-level rise (per m2 ocean)
            for (int n=0; n<nhb; n++) {
                for (int i=1; i<nhb; i++) hrsl[o][n]+=MV[o].getArray()[i][n];
                hrsl[o][n]*=vec[n]; // *v.e.c./cw (layer depths cancel)
                hrsl[o][n]*=frac[o+1]/(frac[1]+frac[2]); //to enable adding the two oceans
            }
            
            //now make the SIUS matrices for changing upwelling
            if (varupwell.istrue()) {
                upwellstep[o]=(MVI[o].times(U.times(MV[o]))).getArray();
                for (int n=0; n<nhb; n++) for (int m=0; m<nhb; m++) upwellstep[o][n][m]*=stepf[m]/val(zeroupwelltemp);
                //check is this the right way round?? was stepf[n] before
            }
            
        } //end for each ocean loop
        
        //don't need these any more -save memory
        //M=null; //MVI=null; MV=null; rate=null; //we do need these if changing upwelling!
        //ED=null;
        Eig=null; rampf=null; stepf=null; hicML=null;
        
    } //end setupfluxes
    
    //****************************
    //reset the boxes according to startyear
    
    public void startstate(int startyear) {
        this.startyear=startyear;
        
        if (startyear==1750) {
            nstd=0; nstdold=0;
            for (int o=0; o<2; o++) {
                qinold[o]=spaceflux[o]*qpt*tstart;
                for (int n=0; n<nhb; n++) hiq[o][n]=hiqstart[o][n];
            }
            
            //checked hiqstart[o][n] givesw correct temps inc 18 at surface
        } //end 1750
        
        //Note the 1999 data is set by running it from 1750 initially
        else {	startyear=2000; //just in case it isn't!
        nstd=nstd99; nstdold=nstdold99; qinold[0]=qinold99[0]; qinold[1]=qinold99[1];
        for (int o=0; o<2; o++) {	for (int n=0; n<nhb; n++) hiq[o][n]=hiq99[o][n]; }
        } //end 2000
        
    } //end startyear
    
    
    //save 1999 state for adjusting future only -called from time loop
    public void save99() {	for (int o=0; o<2; o++) {	nstd99=nstd; nstdold99=nstdold; qinold99[o]=qinold[o]; for (int n=0; n<nhb; n++) hiq99[o][n]=hiq[o][n]; }}
    
    //**************************************
    //this is called from glotemp calcstep when forcing changes
    
    public float adjust(float[] rf) {
        
        //assuming dt=1 for the moment
        //note o+1 gives ocean frac, o*3 gives neighboring land!
        
        //assume nstd increases same as last step
        guess=nstd+(nstd-nstdold); nstd=guess; nstdold=nstd;
        
        for (int o=0; o<2; o++) {	//for each ocean
            
            //apply prop and step functions to oceans
            for (int n=0; n<nhb; n++) {	hiq[o][n]= hpropf[o][n]*hiq[o][n] + shicML[o][n]*qinold[o]; }
            
            //calc heat input to each ocean mixed layer (watt-year per m2)
            //direct input from rf
            qinbase[o] = rf[o+1];
            //add input via land
            qinbase[o] += rf[o*3]*(frac[o*3]/frac[o+1])*klo/(kls*frac[o*3]+klo);
            
            //correct for steady-state flux at starting temp
            qinbase[o] +=spaceflux[o]*qpt*tstart;
            
        }// end o
        
        //now iterate to find north-south temp diff
        nit=0; do {
            
            for (int o=0; o<2; o++) {
                //subtract north-south flux from qin
                qin[o] =qinbase[o] +(o==0 ? -1.0 : 1.0) * nstd * kns / frac[o+1];
                
                dqin=qin[o]-qinold[o];
                
                //apply ramp function and calc new surface temperature
                mlt[o]=0;
                for (int n=0; n<nhb; n++) {	hiqi[o][n]=hiq[o][n] + rhicML[o][n]*dqin; mlt[o]+=hrML[o][n]*hiqi[o][n]; }
                mlt[o]/=qpt; //convert units to temp
            } //end o loop
            
            //now calculate real nstd
            diff=(mlt[0]-mlt[1])*cice -nstd; nstd+=diff;
            
            //keep iterating until negligible temp diff
            nit++; } while(Math.abs(diff)>0.001 && nit<10);
        
        hiq=hiqi;
        qinold=qin;
        
        //correct for start temp converts mlt to change not total
        for (int o=0; o<2; o++) mlt[o]-=tstart;
        
        //store arrays of "output" data for plotting
        curve[] bt=gm(glotemp.class).boxtempqt;
        bt[0].set(year, (float)((mlt[0]*cice*klo + frac[0]*rf[0])/(kls*frac[0]+klo)));
        bt[3].set(year, (float)((mlt[1]*cice*klo + frac[3]*rf[3])/(kls*frac[3]+klo)));
        bt[1].set(year, (float)(mlt[0]*cice));
        bt[2].set(year, (float)(mlt[1]*cice));
        
        float globavtemp=0.0f; for (int i=0; i<4; i++) globavtemp+=(float)(	bt[i].get(year)*frac[i]);
        return globavtemp;
    } //end adjust
    
    
    //*******  sea-level rise due to thermal expansion **********
    public double sealevteinit;
    float  thermalexpansion() {
        float thermexp=0; for (int o=0; o<2; o++) for (int n=0; n<nhb; n++) thermexp+=hrsl[o][n]*hiq[o][n];
        //subtract initial value
        if (year==gsy) sealevteinit=thermexp;
        thermexp-=sealevteinit;
        return thermexp;
    }
    
    //******* temperature- upwelling feedback******
    void  tempupwellfb(double ut) {
        if (varupwell.istrue()) {
            /*
            note call from glotemp calcstep adjusted jan09 to prevent occasional crashes (see notes from sci-fix.todo)
            note:  this is calculated *after* the rest, i.e. affects next years Q -not iterated but should not precede calc sealevel thermexp in calcstep loop!
            beware slow -whole matrix multiplication
            note upwellreducefrac following IPCC TAR WG1 appx9.1 "the proportion of upwelling which is scaled..."
            */
            if (ut>val(zeroupwelltemp)) ut=val(zeroupwelltemp);
            for (int o=0; o<2; o++) for (int n=0; n<nhb; n++) {
                hiquc=0; for (int m=0; m<nhb; m++) hiquc+=hiq[o][m]*upwellstep[o][n][m];
                hiq[o][n]-=hiquc*ut*val(upwellreducefrac);
                //note minus because this is decreasing upwelling!
            }
        } //end if varupwell
    }
    
} //end class


/*
 
        //IPCC AR4 GCMs,  8, 1, 1 are default as specified in FOD Section 8.8.2
        gcm(double a, double b, double c, double d) { this( a, b, c, d, 8.0, 1.0); }
        //IPCC TAR GCMs - note f repeated as kns and klo always same
        gcm(double a, double b, double c, double d, double e, double f) { this( a, b, c, d, e, f, f, 60.0, 0.2, 1.25); }
        //IPCC SAR - also changing mld, psi and seaice, note upwellreducefrac and baserate never change
        gcm(double a, double b, double c, double d, double e, double f, double g, double h, double i, double j) { this( a, b, c, d, e, f, g, h, i, j , 0.3, 4.0); }
        //full constructor,
        gcm(double ...  x) { 	    cp[] c=cp.values();  for (int i=0; i<x.length; i++) val.put(c[i], x[i]); }
 
 enum gcmset {
        //note: scope to  add more sets here - eg ordered by CS or diffusivity
        AR4_all19(EnumSet.range(CCS_M3, AR4_average)),
        AR4_selection(new gcm[]{ECHAM5_MPIOM, MIROC3_2_MED, HadCM3, GISS_EH, HadGEM1, CSIRO_Mk3}),
        TAR_all7(EnumSet.range(GFDL_TAR, DOE)),
        all_GCMs(EnumSet.allOf(gcm.class));
 
        gcm[]list;
        gcmset(EnumSet<gcm> s) { list=s.toArray(new gcm[0]); }
        gcmset(gcm[] g) { list=g;}
    }
 
 
 */

