/*
Shares module calculates regional emissions quotas (for a policy scenario) according to various sharing-schemes
Currently most rules apply only to fossil CO2, needs to be extended to other gases (which are simply scaled by sres baseline)
 */
package jcm.mod.regemit;

import java.util.*;
import jcm.core.*;
import jcm.core.cur.*;
import jcm.core.ob.loopcalc;
import jcm.core.par.param;
import jcm.core.ob.module;
import jcm.core.reg.*;
import jcm.mod.obj.regset;
import jcm.mod.socio.*;
import jcm.mod.resp.*;
import jcm.mod.obj.controller;
import jcm.mod.carbon.carboncycle;
import jcm.mod.luc.futureLUC;
import jcm.mod.obj.globco2emit;
import jcm.mod.ogas.othgasemit;
import static jcm.core.complexity.*;
import static jcm.core.report.*;
import static jcm.gui.gen.colfont.*;
import static jcm.mod.regemit.shares.distriboptions.*;
import static jcm.mod.regemit.shares.convcrit.*;
import static jcm.mod.regemit.shares.igroup.*;
import static jcm.mod.regemit.shares.emitthreshold.*;
import static jcm.mod.regemit.shares.intensity.*;

public class shares extends module {    //
    // =========== ENUMS ============
    enum distriboptions {
        sresdist, convergence, grandfather, responsibility, leastcost, unspecified
    };

    enum convcrit {
        emissionspercapita, emissionspergdp
    };

    enum igroup {
        all_from_start, Kyoto_Annex_B, none
    };

    enum emitthreshold {
        worldemitpercap, axbemitpercap, nolim
    };

    enum intensity {
        redint, redrelbase, base
    };

    //================= CURVES ==============================
    public curveset //
            emitfosquota = new curveset("emitfosquota", "mega&ton&carbon", simplest), //
             emitlucquota = new curveset("emitlucquota", "mega&ton&carbon"), //
             emitch4quota = new curveset("emitch4quota", "mega&ton&CH4"), //
             emitn2oquota = new curveset("emitn2oquota", "mega&ton&N"), //
             emitfosabate, //
             emitfosquotaCO2 = new curveset("emitfosquotaCO2", "mega&ton&CO2", 1890, 2100, expert), //
             emitequiv = new curveset("CO2 Equivalent  (GWP) Emissions", "mega&ton&CO2", 1890, 2100, expert);    //
    //
    //================ PARAMS ===============================
    //
    complexity[] distribcomplex = new complexity[]{simplest, simplest, normal, expert, experimental, experimental};
    public param<distriboptions> distribution = new param("distribmenu", distriboptions.values(), convergence, distribcomplex);
    public param<igroup> initgroup = new param("initial group", igroup.values(), Kyoto_Annex_B, normal);
    public param<emitthreshold> emitlim = new param("emissions threshold", emitthreshold.values(), axbemitpercap, normal);
    public param<convcrit> conv_criteria = new param("convergence criteria", convcrit.values(), emissionspercapita, expert);
    public param<intensity> midincaction = new param("middle-income early action", intensity.values(), base); //
    //
    public param //
            richjoin = new param("join if high GDP", true), //
             axb_gdp_threshold = new param("upper_GDP_threshold", "dollar&per&person", 20000, 10000, 30000), //
             convergey = new param("convyear", "", 2060, 2010, 2100, simplest), //
             exponential = new param("expconvopt", false, expert), //
             cvf = new param("convfac", "", 6, 1, 13, expert), //
             popcoyoption = new param("popcoyopt", false, expert), //  
             popcoy = new param("popcoy", "", 2030, 1990, 2100, expert), // 
             lc_iterative = new param("iterative", false, experimental), //
            //kyotop = new param("kyotoopt", false, experimental), //
            // incusa = new param("incusaopt", true, experimental), //
             ldc_gdp_threshold = new param("lower_GDP_threshold", "dollar&per&person", 5000, 0, 20000), //`
             dc_reduction = new param("midinc intensity reduction", "percent&per&yr", 0.5, 0, 5), //
             rate_caps = new param("rate caps", false, expert), // 
             rate_constraint = new param("max decline rate", "percent&per&yr", 6, 2, 20), //
             d2rate_constraint = new param("max deceleration", "percent&per&yr", 3, 0.5, 5), //
             checkscaling = new param("topdownscale", true, experimental), //use this to test if the final topdownscale makes a difference - it shouldn't
             logparticip = new param("record_transitions", false); //set true to see participation threshold changes in log 
    ;
    //
    //==================== INTERACTIONS ===========
    //
    public void initsetup() {
        regemit = gm(emitbase.class);
        attr = gm(attribTracer.class);
        resp = gm(responsibility.class);
        socio = gm(popgdp.class);
//
        history.setaffectedby(gm(regset.class).regions);
        history.follows(gm(emitbase.class).history);
        //note: history is not affectedby anything else, including params here

        follows(socio);  // per capita etc. - although not always true
        follows(regemit);  // baseline emissions
        follows(globco2emit.class);
        follows(othgasemit.class);
        follows(gm(AviaShipEmit.class).emitmitig); // as the mitigated bunker will affect emissions available to regions
        follows(gm(futureLUC.class).quota); // 

        distribution.setaffectedby(gm(controller.class).objective);
        setaffectedby(gm(controller.class).objective);

        emitfosabate = new curvar(curve.Type.difference, gm(emitbase.class).emitfosbase, emitfosquota);
        emitfosabate.mycomplexity = expert;
        emitfosquota.addOb(new curvar(curve.Type.ratio, emitfosquota, gm(popgdp.class).pop, emitlucquota));
        emitequiv.addOb(new curvar(curve.Type.ratio, emitequiv, gm(popgdp.class).pop));

        //====== set order of params in tree ======
        distribution.priority = 1.3;
        initgroup.priority = 1.23;
        emitlim.priority = 1.22;
        richjoin.priority = 1.21;
        midincaction.priority = 1.20;
        rate_caps.priority = 1.19;
    //   
    //====== arrow controls ======
    //.associate(convergey); //this needs to be a derived qtset, therefore how to make it (unless permanently in regemit?)
    //.associate(popcoy);
    }

    public void setinteractions() { //changing interactions (permanent in initsetup)

        distriboptions dc = distribution.chosen;
        boolean incpart = policyscenario && initgroup.chosen != all_from_start;
        setaffectedby(gm(attribTracer.class), policyscenario && dc == responsibility);
        setaffectedby(gm(costs.class), policyscenario && dc == leastcost);
        setaffectedby(distribution, policyscenario);
        setaffectedby(initgroup, policyscenario);
        setaffectedby(emitlim, incpart);
        setaffectedby(conv_criteria, policyscenario && dc == convergence);
        setaffectedby(convergey, policyscenario && dc == convergence);
        setaffectedby(popcoyoption, policyscenario && dc == convergence && conv_criteria.chosen == emissionspercapita);
        setaffectedby(popcoy, policyscenario && dc == convergence && conv_criteria.chosen == emissionspercapita && popcoyoption.istrue());
        setaffectedby(exponential, policyscenario && dc == convergence);
        setaffectedby(cvf, policyscenario && dc == convergence && exponential.istrue());
        //setaffectedby(incusa, policyscenario && kyotop.istrue());
        setaffectedby(lc_iterative, policyscenario && dc == leastcost);
        setaffectedby(richjoin, incpart);
        setaffectedby(axb_gdp_threshold, incpart && richjoin.istrue());
        setaffectedby(midincaction, incpart);
        setaffectedby(rate_caps, policyscenario);
        setaffectedby(rate_constraint, policyscenario && rate_caps.istrue());
        setaffectedby(d2rate_constraint, policyscenario && rate_caps.istrue());
        setaffectedby(ldc_gdp_threshold, incpart && (midincaction.chosen != base || rate_caps.istrue()));
        setaffectedby(dc_reduction, incpart && midincaction.chosen != base);
    }    //
    //***************** WORKING VARIABLES **********************************
    emitbase regemit;
    attribTracer attr;
    responsibility resp;
    popgdp socio;
    curveset efq, efb;
    List<region> reg_all, reg_allnb = new ArrayList();
    Set<region> particip = new HashSet(), notparticip = new HashSet(), capped = new HashSet();
    Map<region, Float> mac = new HashMap(), abateshare = new HashMap();
    Map<region, Integer> midinc_sy = new HashMap();
    double cf;
    float participtot;
    boolean policyscenario = true;    //
    //
    // *********** HISTORY ************
    // note history methods only run when regions or regemit history changes!
    public loopcalc history = new loopcalc("shares-history") {
        public void precalc() {
            gm(regset.class).clearoldregions(this);
            //needed for mappings of emitfosabate, -  maybe not most efficient method - and make sure done *after* emitfosbase...
            region regset = (region) gm(regset.class).regions.chosen;
            for (region r : regset.reg) {
                emitfosquota.getOrAddCurve(r);
                emitfosabate.getOrAddCurve(r);
            }
            emitfosquota.getOrAddCurve("bunker").color = grey;
        }

        public void calcstep() {
            if (year < fsyfos) {
                settobase(true);
                calctotandequiv();
            }
        }
    }; //history 
    //

//
// ========== basic scaling methods - both history and future ==========
    void settobase(boolean incbunk) {
        //first fill the emitquota to be same as baseline - for both history and future
        for (region r : reg_allnb) {
            efq.set(r, efb.get(r));
            if (incbunk) efq.set("bunker", efb.get("bunker"));
            emitch4quota.set(r, gm(emitbase.class).emitch4base.get(r));
            emitn2oquota.set(r, gm(emitbase.class).emitn2obase.get(r));
        }
    }

    void calctotandequiv() {
        efq.calctot();
        if (year >= 1890 && year <= 2100) { //note year range because of lack of other gas data
            calcequiv();
            emitch4quota.calctot();
            emitn2oquota.calctot();
        }
    }
    //
    //**************** PRE CALC (future) ***********************************
    public void precalc() {
        efq = emitfosquota;
        efb = regemit.emitfosbase;
        cf = cvf.getval() / (double) (convergey.getval() - fsyfos);
        //moved from objective - prob if affects other modules before this in loop?
        policyscenario = !(gm(controller.class).objective.chosen == "nopolicy");
        initregions();
    }

    //**************** MAIN CALC LOOP (future)  ***********************************
    public void calcstep() {
        if (year >= fsyfos) {
            settobase(false);
// note: exclude bunker, this is now set in AviaShipEmit future-mitigated loopcalc which must be calculated *before* shares

            //maybe only need if (year<sy || distribution.chosen=="sresdist") but skipping this causes some problems with convergence (depending on sequence of parameter changes)
            if (policyscenario) {
                setparticipate();
                developing_bottomup();
                distriboptions dc = distribution.chosen;
                if (dc == grandfather) for (region r : particip) efq.set(r, efq.get(r, year - 1));
                if (dc == sresdist) for (region r : particip) efq.set(r, efb.get(r)); //shouldn't be necessary
                if (dc == convergence) converge();
                if (dc == leastcost) leastcost();
                if (dc == responsibility) responsibility();
                scaleandcapparticip();
            }
            if (checkscaling.istrue()) {
                /* now topdownscale converts all to correct global total
                true=exclude bunker (which was scaled in AviaShipEmit)
                if (distribution.chosen!=brazilian) - this exclusion should't make a difference - check! P3 
                 */
                efq.topdownscale(gm(carboncycle.class).fossil.get(), true);
                efq.calctot();
            }
            emitch4quota.topdownscale(gm(othgasemit.class).ch4emit.get());
            emitn2oquota.topdownscale(gm(othgasemit.class).n2oemit.get());
            calctotandequiv();

        } //future
    } //end calcstep

    //******** INCREASING PARTICIPATION **********
    /* Set up initial participants lists
     * 
     */
    void initregions() {
        if (logparticip.istrue()) deb("\n ===========================");
        reg_all =
                ((region) (gm(regset.class).regions.chosen)).reg;
        reg_allnb.clear();
        reg_allnb.addAll(reg_all);
        reg_allnb.remove(
                "bunker");
        particip.clear();
        notparticip.clear();
        midinc_sy.clear();
        mac.clear();
        abateshare.clear();  //probably not necessary - check leastcost code
        if (initgroup.chosen == all_from_start) particip.addAll(reg_allnb);
        else {
            notparticip.addAll(reg_allnb);
            if (initgroup.chosen == Kyoto_Annex_B) makeKyotoAxB();
        }
    }

    void makeKyotoAxB() {
        //note just one nation in AXB is enough to classify whole region! 
        Set<region> A1nat = regman.allreg.find("AXB").subreg(regman.nations); //set of Nartions
        Set<region> AXB = new HashSet(); //set within the chosen regset
        String initparticip = "\nInitial Annex B: ";
        checkreg:
        for (region r : reg_allnb) {
            Set<region> rn = r.subreg(regman.nations);

            for (region rr : rn) if (A1nat.contains(rr)) {
                    AXB.add(r);
                    initparticip +=
                            ", " + r.getName();
                    continue checkreg;
                }

        }
        if (logparticip.istrue()) deb(initparticip);
        notparticip.removeAll(AXB);
        particip.addAll(AXB);
    }

    /*
    Work out which participate - based on previous years data (easier to computer and more realistic policy)
    Note the result is also dependent on the regional resolution used, due to the effect of participation criteria applied to regional average
     */
    void setparticipate() {
        float poptot = 0, fostot = 0, fracpop, fracemit;
        if (notparticip.isEmpty()) return;
        //participate if emissions / capita (last year) > average
        if (emitlim.chosen != nolim) {
            Collection<region> group = (emitlim.chosen == worldemitpercap ? reg_allnb : particip);
            for (region r : group) {
                poptot += socio.pop.get(r, year - 1);
                fostot +=
                        efq.get(r, year - 1);
            }

            for (region r : notparticip)
                if (efq.get(r, year - 1) / socio.pop.get(r, year - 1) > fostot / poptot)
                    changecat(r, " emit/cap > avg (" + fostot / poptot + ")");
        }

        for (region r : notparticip) {
            float gdppercap = socio.gdp_ppp.get(r) / socio.pop.get(r);
            //participate if pass gdp/cap threshold
            if (richjoin.istrue() && gdppercap > axb_gdp_threshold.getval()) changecat(r, " GDP/cap > " + axb_gdp_threshold.getval());
            //if country leaves LDC category, start counting years for cumulative reduction
            if (!midinc_sy.containsKey(r) && gdppercap > ldc_gdp_threshold.getval()) midinc_sy.put(r, 0);
        }

        notparticip.removeAll(particip); //can(t include in loop => concurrent modification error
        if (logparticip.istrue() && notparticip.isEmpty()) deb("\n ============================\n");
    //midinc_sy.keySet().removeAll(particip); //keep to avoid sudden jump up
    }

    void changecat(region r, String reason) {
        particip.add(r);
        if (logparticip.istrue()) deb(r.getName() + "\t\t  joins " + year + "\t  " + reason);
    }

    void developing_bottomup() {
        /* reduce emissions of dc that are in the middle income group (gdp/cap> lower threshold, not yet particpants)
         * option redrelbase: reduce emissions (and hence intensity) by  x%/yr wrt baseline
         * note this reduction should continue even after they join particip group (otherwise sudden jump up!), but the year- gap doesn't keep increasing
         * option redint: reduce emissions intensity by x%/yr (absolute) - but only apply if less than baseline
         * note efq first set to efb in calcstep
         */
        if (midincaction.chosen != base) {
            for (region r : midinc_sy.keySet()) {
                int yd = midinc_sy.get(r);
                if (notparticip.contains(r)) {
                    yd++;
                    midinc_sy.put(r, yd);
                }

                float fracreduce = (float) Math.pow((1.0 - dc_reduction.getval() / 100f), yd);
                if (midincaction.chosen == redrelbase && yd > 0) efq.set(r, (efq.get(r) * fracreduce));
                if (midincaction.chosen == redint && yd > 0) {
                    float intensity = (efq.get(r, (year - yd)) / socio.gdp_ppp.get(r, (year - yd))) * fracreduce;
                    efq.set(r, Math.min(intensity * socio.gdp_ppp.get(r), efq.get(r)));
                }

            }
        }
        //now add up all DC emissions to calculate what's left for the particip group
        float emittot = 0;
        for (region r : notparticip) emittot += efq.get(r);
        participtot =
                gm(carboncycle.class).fossil.get() - efq.get("bunker") - emittot;
    }

    void scaleandcapparticip() {
        float excess = scaleandcap(particip, participtot);
        /* 
        if excess <0, it's because total after capping is greater than available to the participating group
        by this stage, all must be capped, we have to reduce more in the medium-income DCs
        P3 in theory a better alternative here would be to exceed the global target temporarily - but that could break the stabilisation iteration
         */
        if (excess < 0) {
            float midinctot = 0;
            Set<region> midinc = new HashSet(midinc_sy.keySet());
            midinc.removeAll(particip);
            for (region r : midinc) midinctot += efq.get(r);
            excess =
                    scaleandcap(midinc, midinctot + excess);
        }
        /*
        if excess still <0 have really run out of emissions space, have to scale all down!
         */

        if (excess < 0) {
            float alltot = 0;
            for (region r : reg_allnb) alltot += efq.get(r);
            float fac = (gm(carboncycle.class).fossil.get() - efq.get("bunker")) / alltot;
            for (region r : reg_allnb) efq.set(r, efq.get(r) * fac);
        }
    }

    float scaleandcap(Set<region> group, float grouptot) {
        boolean somechange;
        float emittot, rate, oldrate, fac;
        capped.clear();
        float rc = (float) rate_constraint.getval() / 100f;
        float d2rc = (float) d2rate_constraint.getval() / 100f;

        //apply caps
        do {
            somechange = false;
            emittot =
                    0;
            for (region r : group) if (!capped.contains(r)) emittot += efq.get(r);
            fac =
                    grouptot / emittot;
            //scale to fit within total
            for (region r : group) if (!capped.contains(r)) efq.set(r, efq.get(r) * fac);
            if (!(rate_caps.istrue()) || year < fsyfos + 10) return 0; //ignore caps at beginning - due to initial fluctuations
            //apply rate constraints
            checkreg:
            for (region r : group) if (!capped.contains(r)) {
                    rate = efq.get(r, year) - efq.get(r, year - 1);
                    oldrate =
                            efq.get(r, year - 1) - efq.get(r, year - 2);

                    //deceleration rate cap - note this can cause problems as countries hit bottom
                    if (rate < 0 && (rate - oldrate) / efq.get(r) < -d2rc) {
                        efq.set(r, efq.get(r, year - 1) * (1f - d2rc) + oldrate);
                        somechange =
                                true;
                    }

//max decline rate cap
                    rate = efq.get(r, year) - efq.get(r, year - 1);
                    if (rate / efq.get(r) < -rc) {
                        efq.set(r, efq.get(r, year - 1) * (1f - rc));
                        somechange =
                                true;
                    }

                    if (somechange) {
                        capped.add(r);
                        if (efq.get(r) < 0) efq.set(r, 0);
                        if (efq.get(r) > efb.get(r)) efq.set(r, efb.get(r)); // shouldn't be necessary!
                        grouptot -=
                                efq.get(r);
                    }

                }
        } while (somechange);
        return grouptot;
    }
//******** CONVERGENCE *****************************
    void converge() {
//note this sets a fractional share, normalising to correct total done later by topdownscale
        int cyear = (conv_criteria.chosen == emissionspercapita && year > popcoy.getval() && popcoyoption.istrue()) ? (int) (popcoy.getval()) : year;

        double f =
                (year >= convergey.getval()) ? 1
                : (exponential.istrue()) ? (double) Math.exp(cf * (year - convergey.getval() - 1))
                : (double) 1.0 / (convergey.getval() - year);

        float poptot = 0, gdptot = 0, emittot = 0; //note if implement  a standard world total region for all regional data, can use this instead

        for (region r : particip) {
            poptot += socio.pop.get(r, cyear);
            gdptot +=
                    socio.gdp_ppp.get(r, cyear);
            emittot +=
                    efq.get(r);
        }
//main convergence formula
        float cr = 1, er;
        for (region r : particip) {
            if (conv_criteria.chosen == emissionspercapita) cr = (float) socio.pop.get(r, cyear) / poptot;
            if (conv_criteria.chosen == emissionspergdp) cr = (float) socio.gdp_ppp.get(r, cyear) / gdptot;
            er =
                    efq.get(r, year - 1) / emittot;
            efq.set(r, (float) (er - f * (er - cr)));
        }

    } //end convergence

    //********** RESPONSIBILITY ****************
    /*
    Based on Brazilian proposal
    Reduction in each year is proportional to share of cumulative responsibility
    
    Note responsibility endyear should be set to 2100!
    Otherwise the relative increase in DC responsibility isn't considered    
     */
    void responsibility() {

        float tempinctot = 0, tempincp = 0, oldemittot = 0, totreduce;

        int respyear = (int) Math.min(year - 1, resp.endyear.getval());

        for (region r : particip) if (efq.get(r, year - 1) > 0) {
                tempinctot += attr.surftemp.get(r, respyear);
                oldemittot +=
                        efq.get(r, year - 1);
            }

        totreduce = oldemittot - participtot;
        if (totreduce < 0) {
            deb("Problem applying Brazilian proposal: total reduction positive in " + year);
            return;
        }
        for (region r : particip) {

            float fr = attr.surftemp.get(r, respyear) / tempinctot; //fraction responsibility
            efq.set(r, efq.get(r, year - 1) - totreduce * fr);
            if (efq.get(r, year) < 0) efq.set(r, 0);
        //if (efq.get(r,year)>efb.get(r,year)) efq.set(r, efb.get(r,year));
        }

    } //end responsibility


//*********** LEASTCOST ***************************
    void leastcost() {

        /*
        /Probably doesn't work any more - see todo in 
        
        Note: should adjust also by popweighting if want maximise welfare
        
        Principle is to make Marginal cost (=dcost/dabate) same for all regions
        For Nordhaus cost function, differentiating gives:
        marginal cost  = beta  * fac1 * abate ^ (beta-1)
        where beta = abatepow 	and fac1 = gdp*abatelin*b1 / emitbase ^ beta
        =>  (marginal cost / beta) ^ (1/(beta-1)) = fac1 ^ (1 / (beta-1)) * abate = constant fac3
        
        Problem is that formula can give negative emissions to some regions, especially India
        can fix at zero, but then must scale to get right total, maybe no longer least-cost?
        Checking MACs: they are all very close, at least before some regions reach zero
        
        For Macgem no algebraic solution because the beta are all different => need to iterate
        (although in practice formula below gives reasonable results)
        Calc macs from previous year, allocate (extra) abatement in inverse proportion to MACs
        Alternative method: Give all abatement to region with lowest mac? (maybe split into smaller chunks)
        or keep adjusting frac of abatement: start at 1, and reduce as mac higher
        Or could assume MACs will change by same amount as previous step?
        
         */

        float suma = 0, sumb = 0, sums = 0, summ = 0;
        int nregs = particip.size();

        if (lc_iterative.istrue()) {
            for (region r : particip) {
                mac.put(r, 1f / (gm(costs.class).mac(r, year - 1) * gm(costs.class).ew(r, year - 1)));

                if (Float.isInfinite(mac.get(r)) || Float.isNaN(mac.get(r))) mac.put(r, 0f);
                summ += mac.get(r);
            }
            for (region r : particip) {
                abateshare.put(r,
                               (summ > 0) ? abateshare.get(r) * (0.6f + 0.4f * mac.get(r) / summ)
                               : efb.get(r) / gm(carboncycle.class).fossil.get());
                efq.set(r, efq.get(r, year - 1) + (efb.get(r) - efb.get(r, year - 1)));
            }
        } else {	//algebraic solution - no weights yet
            for (region r : particip) {
                abateshare.put(r, 1f / (float) Math.pow(gm(costs.class).alpha(r), 1f / (gm(costs.class).beta(r) - 1f)));
                efq.set(r, efb.get(r));
            }
        }
        for (region r : particip) {
            sums += abateshare.get(r);
            sumb +=
                    efq.get(r);
        }//nr
        suma = sumb - gm(carboncycle.class).fossil.get();

        for (region r : particip) {
            efq.set(r, efq.get(r) - suma * abateshare.get(r) / sums);
            if (efq.get(r) < 0) {
                efq.set(r, 0);
            }
            if (efq.get(r) > efb.get(r)) {
                efq.set(r, efb.get(r));
            }
        }
    } //leastcost

//************* EQUIV *****
    /*equivalents using GWP
    created for comparison with attribution of temperature, therefore only including CO2, inc all LUC, CH4 and N2O
    may be extended for other applications later
    GWPs from TAR 100 year time horizon (for 10yr: 62 & 275, for 500yr: 7 & 156)
     *44/12 => mass CO2, *44/14 => mass N2O, note ch4emit was already in MtCH4
     */
    static float gwpch4 = 23,  gwpn2o = 296;

    void calcequiv() { //called from shares  calcstep in order to be at right point of loop

        for (region r : reg_allnb) {
            emitequiv.set(r, (efq.get(r) + emitlucquota.get(r)) * 44f / 12f + emitch4quota.get(r) * gwpch4 + emitn2oquota.get(r) * gwpn2o * 44f / 14f);
            emitfosquotaCO2.set(r, efq.get(r) * 44f / 12f);
        }

        emitequiv.calctot();
        emitfosquotaCO2.calctot();
    }
} //end  class
//*************************************

